/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.api.session;

import org.json.JSONObject;
import com.openexchange.usm.api.exceptions.USMException;

/**
 * Interface that can be provided to the USM system. It specifies a handler that will be called if a session is created for the first time,
 * i.e. if no persistent information is stored in the database. It will be also called on later accesses to update session-internal data
 * from any modified OX configuration and to verify that access to the session is still allowed.This handler can be used to configure the
 * session, e.g. to set which ContentTypes (and which fields) are used by the client. This handler also specifies which optional behavior of
 * the USM sync system should be enabled.
 * 
 * @author afe
 */
public interface SessionInitializer {

    /**
     * Called after a Session is created for the first time, i.e. if no persistent information is found in the database. It is guaranteed
     * that the login for the session already succeeded. The Session is internally already completely initialized, all methods may be
     * called.
     * 
     * @param session
     * @param ajaxConfiguration configuration as retrieved from the OX JSON interface
     * @throws USMException
     */
    void initializeSession(Session session, JSONObject ajaxConfiguration) throws USMException;

    /**
     * Called after a Session has been loaded from its persistent state. This method is not called if the Session is initialized for the
     * first time, in which case initializeSession is called. This method can be used to check configuration parameters that have changed
     * and update the persistent fields accordingly.<br>
     * It is guaranteed that the login for the session already succeeded. The Session is internally already completely initialized, all
     * methods may be called.
     * 
     * @param session
     * @param ajaxConfiguration
     * @throws USMException
     */
    void reinitializeSession(Session session, JSONObject ajaxConfiguration) throws USMException;

    /**
     * Called on regular intervals to make sure the access to the specific protocol is still enabled for that session.
     * 
     * @param session
     * @throws USMException
     */
    void checkAccess(Session session) throws USMException;

    /**
     * Called if the default mail folders have changed.
     * 
     * @param session
     * @param ajaxConfiguration
     * @throws USMException
     */
    void updateDefaultFolders(Session session, JSONObject ajaxConfiguration) throws USMException;

    /**
     * @return true if this protocol expects modified folders to be reported on each waitForChanges()-call. If this method returns false,
     *         the internal modified flag will be reset after the folder has been reported as modified for a few (internally configured)
     *         times
     */
    boolean allowsUnlimitedWaitForChangesOnModifiedFolders();

    /**
     * @return true if this protocols identifies objects by their UUID, false if they are identified by their OX ID. If they are identified
     *         by their UUID, the sync mechanism can implement internal mappings, e.g. if the OX-ID of standard-folders has changed.
     */
    boolean objectsSynchronizedByUUID();
}
