/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.api.datatypes;

import java.io.Serializable;
import java.util.Arrays;
import java.util.UUID;

/**
 * Stores all information on attachments to an OX PIM object, including the last modified timestamp. This class is serializable, only the
 * relevant sync information (timestamp, ox ids and uuids) will be serialized.
 * 
 * @author afe
 */
public class PIMAttachments implements Serializable {

    /**
	 * 
	 */
    private static final long serialVersionUID = 1L;

    private final long _timestamp;

    private final PIMAttachment[] _attachments;

    // Special flag that can be set to force this object to differ from other PIMAttachments (to enforce that a SyncState is written, since
    // normal equals only checks for the timestamp, not for the contained PIMAttachment objects)
    private transient boolean _forceModified;

    public PIMAttachments(long timestamp, PIMAttachment... attachments) {
        this(timestamp, false, attachments);
    }

    public PIMAttachments(long timestamp, boolean forceModified, PIMAttachment... attachments) {
        _timestamp = timestamp;
        _attachments = attachments;
        _forceModified = forceModified;
    }

    public long getTimestamp() {
        return _timestamp;
    }

    public int size() {
        return (_attachments == null) ? 0 : _attachments.length;
    }

    public PIMAttachment[] getAttachments() {
        return _attachments;
    }

    public boolean containsAttachment(UUID uuid) {
        if (uuid != null) {
            for (PIMAttachment a : _attachments) {
                if (uuid.equals(a.getUUID()))
                    return true;
            }
        }
        return false;
    }

    /**
     * @return true if this PIMAttachments is set to be forced as not equals to other PIMAttachments with the same timestamp (normal equals
     *         does not check the contained PIMAttachment objects, only the timestamp)
     */
    public boolean isForceModified() {
        return _forceModified;
    }

    public void setForceModified(boolean forceModified) {
        _forceModified = forceModified;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + (int) (_timestamp ^ (_timestamp >>> 32));
        return result;
    }

    /**
     * Returns true if obj is a PIMAttachments object that has the same timestamp as this one and if neither of the 2 PIMAttachments objects
     * has the forceModified flag set.
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (!(obj instanceof PIMAttachments))
            return false;
        PIMAttachments other = (PIMAttachments) obj;
        if (_timestamp != other._timestamp)
            return false;
        if (_forceModified || other._forceModified)
            return false;
        return true;
    }

    @Override
    public String toString() {
        return "PIMAttachments(timestamp=" + _timestamp + ',' + _forceModified + ',' + Arrays.toString(_attachments) + ')';
    }
}
