/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.api.datatypes;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.session.Session;

/**
 * DataType for fields that store a Number and that always have a value (i.e. never null, 0 as default instead of null)
 * 
 * @author afe
 *
 */
public class NumberNotNullDataType extends AbstractDataType<Number> {
	

    public NumberNotNullDataType() {
        super("NumberNotNull");
    }

	@Override
    public void addToJSONArray(Session session, JSONArray parent, Object object) {
		parent.put(object);
	}

	@Override
    public Number checkValue(Object o) {
		if (o == null)
			return 0;
		if (!(o instanceof Number))
			generateTypeError(o);
		return (Number) o;
	}

	@Override
    public Number extractFromJSONArray(Session session, JSONArray array, int index) throws JSONException {
		if (array.isNull(index))
			return 0;
		return convertToNumber(array.get(index));
	}

	@Override
    public Number extractFromJSONObject(Session session, JSONObject jsonObject, String field) throws JSONException {
		if (jsonObject.isNull(field))
			return 0;
		return convertToNumber(jsonObject.get(field));
	}

	@Override
    public void storeInJSONObject(Session session, JSONObject parent, String field, Object object) throws JSONException {
		parent.put(field, object);
	}

	@Override
	public Number createNewObject() {
		return 0;
	}

	@Override
	public boolean isDefaultValue(Object object) {
		return ((Number) object).intValue() == 0;
	}
}
