/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.api.cache;

import com.openexchange.usm.api.session.assets.CompleteSessionID;

/**
 * {@link SyncStateCacheProvider}. A provider for SyncStateCaches that USM should use (to reduce number of database read operations for
 * stored SyncState data). If no SyncStateCacheProvider is available, USM will use an internal implementation that uses the Java heap to
 * store the SyncStates.
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public interface SyncStateCacheProvider {

    /**
     * Returns a SyncStateCache that can be used for the given USM session referenced by the CompleteSessionID. The SyncStateCacheProvider
     * may return the same cache for all sessions or individual caches for each or some of the sessions. All requests on individual caches
     * will also use the CompleteSessionID.
     * 
     * @param session
     * @return SyncStateCache that should be used for the given USM session.
     */
    SyncStateCache getCache(CompleteSessionID session);

    /**
     * This method is called if the given USM session has been removed from memory (i.e. it is not active for some time, but might be
     * reactivated in the future). The SyncStateCacheProvider may execute any actions to reduce local resource usage, but is not required to
     * do so if enough resources are available.
     * 
     * @param session
     */
    void freeCache(CompleteSessionID session);

    /**
     * This method is called if the given USM session has been completely removed from the system (i.e. no future calls to this session are
     * to be expected, the client has terminated any synchronization). The SyncStateCacheProvider should remove any cached data for that
     * session. Note that this method may never get called, so the SyncStateCacheProvider should on its own remove any data that is very old
     * and has not been accessed in a very long time. (any cached data can be retrieved from the database as long as the session is stored
     * there)
     * 
     * @param session
     */
    void removeCacheCompletely(CompleteSessionID session);

    /**
     * @return the current number of SyncStates stored in all caches of all sessions. Used by USM to provide cache information in the JMX
     *         interface.
     */
    int getSyncStateCount();

    /**
     * Called to remove any cached data that is older than the configured limit for cached sync states. Any sync state which has not been
     * accessed since the given limit should be removed from the cache (special cache implementations may choose to ignore this if they have
     * enough resources available).
     * 
     * @param limit timestamp (in milliseconds since the epoch). All sync states that have not been accessed since that time should be
     *            removed
     * @return total number of sync states that have been removed from all sync state caches
     */
    int removeUnusedSyncStates(long limit);
}
