/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.util;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import com.openexchange.exception.OXException;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.usm.api.io.InputStreamProvider;

/**
 * {@link TempFile} wrapper that updates associated ManagedFile on every write to a temporary file.
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class TempFile implements Closeable, InputStreamProvider {

    private final ManagedFile _managedFile;

    private final RandomAccessFile _randomAccessFile;

    public TempFile(ManagedFile managedFile, long offset) throws IOException {
        _managedFile = managedFile;
        File f = managedFile.getFile();
        if (f == null)
            throw new IOException("Temp file no longer available");
        _randomAccessFile = new RandomAccessFile(f, "rw");
        seek(offset);
    }

    public TempFile(ManagedFile managedFile) {
        _managedFile = managedFile;
        _randomAccessFile = null;
    }

    public TempFile(File file, long offset) throws IOException {
        _managedFile = null;
        _randomAccessFile = new RandomAccessFile(file, "rw");
        seek(offset);
    }

    public TempFile(File file) throws FileNotFoundException {
        _managedFile = null;
        _randomAccessFile = new RandomAccessFile(file, "r");
    }

    public String getFileName() {
        String fileName = "";
        if (_managedFile != null)
            fileName = _managedFile.getFile().getName();
        return fileName;
    }

    private void seek(long offset) throws IOException {
        try {
            _randomAccessFile.seek(offset);
        } catch (IOException e) {
            Toolkit.close(_randomAccessFile);
            throw e;
        }
    }

    public void write(byte[] data) throws IOException {
        write(data, 0, data.length);
    }

    /**
     * Writes <code>len</code> bytes from the specified byte array
     * starting at offset <code>off</code> to this file.
     *
     * @param      data   the data.
     * @param      offset the start offset in the data.
     * @param      len    the number of bytes to write.
     * @exception  IOException  if an I/O error occurs.
     */
    public void write(byte[] data, int offset, int len) throws IOException {
        if (_randomAccessFile == null)
            throw new IOException("Can not access TempFile: Not open for write access");
        _randomAccessFile.write(data, offset, len);
        if (_managedFile != null)
            _managedFile.touch();
    }

    @Override
    public void close() throws IOException {
        if (_randomAccessFile != null) {
            try {
                if (_managedFile != null)
                    _managedFile.setSize(_randomAccessFile.length());
            } finally {
                _randomAccessFile.close();
            }
        }
    }

    public long getSize() throws IOException {
        if (_managedFile != null)
            return _managedFile.getSize();
        if (_randomAccessFile != null)
            return _randomAccessFile.length();
        return 0L;
    }

    @Override
    public InputStream getInputStream() throws IOException {
        if (_managedFile != null) {
            try {
                return _managedFile.getInputStream();
            } catch (OXException e) {
                throw new IOException("Can't access InputStream of ManagedFile", e);
            }
        }
        if (_randomAccessFile == null)
            throw new IOException("File is not accessible");
        _randomAccessFile.seek(0L);
        return new FileInputStream(_randomAccessFile.getFD());
    }
}
