/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.util;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import com.openexchange.usm.api.session.Session;

/**
 * This class stores all information necessary to determine if a client behaves badly, i.e. it performs too many unnecessary synchronizations.
 * It is used and managed by AccessLimiter.
 * 
 * @author afe
 *
 */
public class AccessCounter {

	private static final String DATE_FORMAT_STRING = "yyyy-MM-dd HH:mm:ss.SSS Z";

	private final AccessLimiter _limiter;

	private final long[] _unnecessaryRequests;

	private int _currentIndex;
	private long _blockedUntil;

	public AccessCounter(AccessLimiter limiter) {
		_limiter = limiter;
		_unnecessaryRequests = new long[limiter.getMaxAccesses()];
	}

	/**
	 * Checks whether access is currently denied for this USM session. It will also remove this AccessCounter if the last bad request
	 * is outside the check interval (i.e. stops the counting since there was no bad access within the test interval).
	 * 
	 * @param session
	 * @return true if access should be denied, false if access may be granted
	 */
	public boolean isAccessDenied(Session session) {
		long now = System.currentTimeMillis();
		if (_blockedUntil >= now)
			return true;
		if (_unnecessaryRequests[_currentIndex] < now - _limiter.getTestInterval())
			_limiter.removeCounter(session);
		return false;
	}

	/**
	 * This should be called when an unnecessary access is detected. It will check if there were too many bad accesses within the
	 * test interval, and if true, it will log a warning and enable blocking of all future requests (i.e. isAccessDenied() will return true)
	 * for the configured refusal interval.
	 * 
	 * @param session
	 */
	public void onUnnecessaryAccess(Session session) {
		long now = System.currentTimeMillis();
		if (_blockedUntil >= now)
			return;
		if (_currentIndex >= _unnecessaryRequests.length - 1)
			_currentIndex = 0;
		else
			_currentIndex++;
		long difference = now - _unnecessaryRequests[_currentIndex];
		if (difference < _limiter.getTestInterval()) {
			_blockedUntil = now + _limiter.getRefusalInterval();
			_limiter.getJournal().warn(
					session + " Too many unnecessary synchronizations (" + _limiter.getMaxAccesses() + " in "
							+ difference + " ms.), access will be blocked until "
							+ new SimpleDateFormat(DATE_FORMAT_STRING, Locale.ENGLISH).format(new Date(_blockedUntil)));
		}
		_unnecessaryRequests[_currentIndex] = now;
	}
}
