/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.util;

import java.util.ArrayList;
import java.util.Dictionary;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceEvent;
import org.osgi.framework.ServiceListener;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.osgi.console.ServiceStateLookup;

/**
 * {@link AbstractUSMActivator}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public abstract class AbstractUSMActivator implements BundleActivator {

    protected static final Logger LOG = LoggerFactory.getLogger(AbstractUSMActivator.class);

    protected final Class<?> neededServices[];

    protected final Map<Class<?>, ServiceReference<?>> presentServices;

    private final Map<Class<?>, ServiceRegistration<?>> serviceRegistrations;

    protected BundleContext context;

    /**
     * Initializes a new {@link AbstractUSMActivator}.
     */
    public AbstractUSMActivator() {
        neededServices = getNeededServices();
        presentServices = new HashMap<Class<?>, ServiceReference<?>>(neededServices.length);
        serviceRegistrations = new HashMap<Class<?>, ServiceRegistration<?>>();
    }

    /*
     * (non-Javadoc)
     * @see org.osgi.framework.BundleActivator#start(org.osgi.framework.BundleContext)
     */
    @Override
    public void start(final BundleContext ctx) throws Exception {
        context = ctx;
        LOG.info("Starting bundle: " + context.getBundle().getSymbolicName());

        if (neededServices.length > 0) {
            String filter = OSGiToolkit.createServiceFilter(neededServices);

            try {
                context.addServiceListener(new USMServiceListener(neededServices), filter);
            } catch (InvalidSyntaxException e) {
                LOG.error("Invalid syntax for needed service filter", e);
            }
            for (Class<?> c : neededServices) {
                ServiceReference<?> ref = context.getServiceReference(c);
                presentServices.put(c, ref);
            }
        }

        setState();

        if (OSGiToolkit.allServicesPresent(presentServices.values())) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("All needed services are present for bundle {}. Proceeding with registration.", context.getBundle().getSymbolicName());
            }
            register();
        } else {
            LOG.debug("Needed services for bundle {} are not yet present. Postponing start-up.\n{}", context.getBundle().getSymbolicName(), getMissingServices());
        }
    }

    /*
     * (non-Javadoc)
     * @see org.osgi.framework.BundleActivator#stop(org.osgi.framework.BundleContext)
     */
    @Override
    public void stop(BundleContext ctx) throws Exception {
        LOG.info("Stopping bundle: " + context.getBundle().getSymbolicName());
        unregisterServices();
        unregister();
    }

    /**
     * Get the specified service
     *
     * @param clazz
     * @return
     */
    protected <S> Object getService(Class<S> clazz) {
        ServiceReference<?> serviceReference = presentServices.get(clazz);
        return (serviceReference == null) ? null : context.getService(serviceReference);
    }

    /**
     * Register a service
     *
     * @param clazz
     * @param service
     */
    protected <S> void registerService(Class<S> clazz, S service) {
        registerService(clazz, service, null);
    }

    /**
     * Register the specified service with the specified properties
     *
     * @param clazz
     * @param service
     * @param dictionary
     */
    protected <S> void registerService(Class<S> clazz, S service, Dictionary<String, ?> dictionary) {
        serviceRegistrations.put(clazz, context.registerService(clazz, service, dictionary));
    }

    /**
     * Unregister the specified service
     *
     * @param service
     */
    protected <S> void unregisterService(S service) {
        ServiceRegistration<?> reg = serviceRegistrations.remove(service);
        if (reg != null) {
            reg.unregister();
        }
    }

    /**
     * Return the service that was registered with registerService.
     *
     * @param clazz
     * @return
     */
    protected <S> Object getRegisteredService(Class<S> clazz) {
        S service = null;
        ServiceReference<S> serviceRef = context.getServiceReference(clazz);
        if (serviceRef != null) {
            service = context.getService(serviceRef);
        }
        return service;
    }

    /**
     * Unregister all services
     */
    protected void unregisterServices() {
        for (ServiceRegistration<?> r : serviceRegistrations.values()) {
            context.ungetService(r.getReference());
            r.unregister();
        }
        serviceRegistrations.clear();
    }

    /**
     * Return an array with all needed services
     *
     * @return
     */
    protected abstract Class<?>[] getNeededServices();

    /**
     * Return an array with all optional services
     *
     * @return
     */
    protected Class<?>[] getOptionalServices() {
        return new Class<?>[] {};
    }

    /**
     * Register the service(s)
     */
    protected abstract void register() throws Exception;

    /**
     * Unregister the service(s)
     */
    protected abstract void unregister() throws Exception;

    /**
     * Used for the missing OSGi command.
     */
    private void setState() {
        ServiceReference<ServiceStateLookup> sr = context.getServiceReference(ServiceStateLookup.class);
        if (sr != null) {
            ServiceStateLookup lookup = context.getService(sr);
            List<String> missing = new ArrayList<String>();
            List<String> present = new ArrayList<String>();
            for (int i = 0; i < neededServices.length; i++) {
                String serviceName = neededServices[i].getName();
                if (presentServices.get(neededServices[i]) == null) {
                    missing.add(serviceName);
                } else {
                    present.add(serviceName);
                }
            }
            lookup.setState(context.getBundle().getSymbolicName(), missing, present);
        }
    }

    /**
     * Get a string representation of the missing services
     *
     * @return
     */
    private String getMissingServices() {
        StringBuilder builder = new StringBuilder();
        builder.append("Missing Services for bundle ").append(context.getBundle().getSymbolicName()).append(": {");
        for (Class<?> c : presentServices.keySet()) {
            if (presentServices.get(c) == null) {
                builder.append("\n\t\t\t").append(c.getName());
            }
        }
        builder.append("}");
        return builder.toString();
    }

    /**
     * {@link USMServiceListener}
     */
    private class USMServiceListener implements ServiceListener {

        private final Class<?>[] neededServices;

        /**
         * Initializes a new {@link AbstractUSMActivator.USMServiceListener}.
         */
        USMServiceListener(Class<?>[] neededServices) {
            super();
            this.neededServices = neededServices;
        }

        /*
         * (non-Javadoc)
         * @see org.osgi.framework.ServiceListener#serviceChanged(org.osgi.framework.ServiceEvent)
         */
        @Override
        public void serviceChanged(ServiceEvent sev) {
            ServiceReference<?> ref = sev.getServiceReference();
            Object service = context.getService(ref);
            Class<?> oclazz = service.getClass().getInterfaces()[0];

            // Determine associated service class
            Class<?> serviceClazz = null;
            for (int i = neededServices.length;  null == serviceClazz && i-- > 0;) {
                Class<?> neededService = neededServices[i];
                if (neededService.isAssignableFrom(oclazz)) {
                    serviceClazz = neededService;
                }
            }

            if (null == serviceClazz) {
                // Huh...?
                context.ungetService(ref);
                return;
            }

            switch (sev.getType()) {

            case ServiceEvent.MODIFIED:
            case ServiceEvent.MODIFIED_ENDMATCH:
            case ServiceEvent.REGISTERED:
                LOG.debug("Needed service {} for bundle {} is registered.", ref, context.getBundle().getSymbolicName());
                presentServices.put(serviceClazz, ref);
                setState();

                if (OSGiToolkit.allServicesPresent(presentServices.values())) {
                    LOG.debug("All needed services are present for bundle {}. Proceeding with registration.", context.getBundle().getSymbolicName());

                    try {
                        register();
                    } catch (Throwable t) {
                        LOG.error("Failed to start bundle: {}", context.getBundle().getSymbolicName(), t);
                        stopBundle();
                    }
                } else {
                    LOG.debug(getMissingServices());
                }
                break;

            case ServiceEvent.UNREGISTERING:
                LOG.debug("Needed service {} for bundle {} was unregistered.", ref, context.getBundle().getSymbolicName());
                presentServices.put(serviceClazz, null);
                setState();
                stopBundle();
                break;

            default:
                break;
            }
        }

        private void stopBundle() {
            try {
                context.getBundle().stop();
            } catch (Exception e) {
                e.printStackTrace();
                LOG.warn("Failed to stop bundle: {}", context.getBundle().getSymbolicName(), e);
            }
        }
    }
}
