/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.session.sync;

import java.lang.ref.SoftReference;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import com.openexchange.usm.api.session.DataObject;

/**
 * Cache that globally stores the object data retrieved from the OX server, to be reused if the client executes
 * multiple syncs in a short time for the same folder without executing any changes (e.g. to retrieve multiple
 * small parts of a big synchronization).
 * 
 * @author afe
 *
 */
public class OXDataCache {
	// 5 minutes as interval to check for removal of dead references in the cache table
	private final static long CHECK_INTERVAL = 300000L;

	// Limit in milliseconds how long cached data is considered valid for complete syncs, 0 == caching is completely disabled
	private final long _cacheTimeLimitComplete;

	// Limit in milliseconds how long cached data is considered valid for incomplete syncs, 0 == caching is completely disabled
	private final long _cacheTimeLimitIncomplete;

	private final Map<OXDataCacheID, SoftReference<CachedOXData>> _cache = new ConcurrentHashMap<OXDataCacheID, SoftReference<CachedOXData>>();

	// Used to check periodically for removal of cache entries where the reference has been cleared
	private volatile long _lastRemovalTime = 0L;

	public OXDataCache(long cacheTimeLimitNormal, long cacheTimeLimitIncomplete) {
		_cacheTimeLimitComplete = cacheTimeLimitNormal;
		_cacheTimeLimitIncomplete = cacheTimeLimitIncomplete;
	}

	public CachedOXData getCachedData(OXDataCacheID id) {
		if (!isCacheEnabled())
			return null;
		removeOldReferences();
		SoftReference<CachedOXData> value = _cache.get(id);
		if (value != null) {
			CachedOXData data = value.get();
			if (isCacheDataValid(data, System.currentTimeMillis()))
				return data;
			_cache.remove(id);
		}
		return null;
	}

	public CachedOXData storeCachedData(OXDataCacheID id, DataObject[] data) {
		if (!isCacheEnabled())
			return null;
		removeOldReferences();
		CachedOXData cachedData = new CachedOXData(System.currentTimeMillis(), data);
		_cache.put(id, new SoftReference<CachedOXData>(cachedData));
		return cachedData;
	}

	public void removeCachedData(OXDataCacheID id) {
		if (isCacheEnabled()) {
			removeOldReferences();
			_cache.remove(id);
		}
	}

	private void removeOldReferences() {
		long now = System.currentTimeMillis();
		synchronized (_cache) {
			if (_lastRemovalTime + CHECK_INTERVAL < now) {
				_lastRemovalTime = now;
				for (Iterator<Entry<OXDataCacheID, SoftReference<CachedOXData>>> i = _cache.entrySet().iterator(); i
						.hasNext();) {
					Entry<OXDataCacheID, SoftReference<CachedOXData>> entry = i.next();
					SoftReference<CachedOXData> val = entry.getValue();
					if (val == null || !isCacheDataValid(val.get(), now))
						i.remove();
				}
			}
		}
	}

	private boolean isCacheDataValid(CachedOXData data, long now) {
		if (data == null)
			return false;
		long limit = data.isSyncComplete() ? _cacheTimeLimitComplete : _cacheTimeLimitIncomplete;
		return limit != 0L && data.getTimestamp() > now - limit;
	}

	private boolean isCacheEnabled() {
		return _cacheTimeLimitComplete != 0L || _cacheTimeLimitIncomplete != 0L;
	}
}
