/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.sync;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.exceptions.ConflictingChangeException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.assets.SyncResult;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.dataobject.DataObjectUtil;
import com.openexchange.usm.session.dataobject.FolderHierarchyComparator;
import com.openexchange.usm.session.impl.SessionManagerImpl;
import com.openexchange.usm.session.impl.USMSessionManagementErrorCodes;

public class ContentSyncerSupport {

    private static final String ID = "id";

    private static final boolean READ_SERVER_CHANGES_ON_CREATION = true;

    private static final boolean READ_SERVER_CHANGES_ON_MODIFICATION = true;

    private static interface UpdateHandler {

        void update(DataObject update, DataObject serverObject, boolean isFolderUpdate) throws USMException;
    }

    public static SyncResult storeDataObjectsAndBuildClientResult(boolean incomplete, ContentSyncerStorage serverInterface, DataObjectSet serverDataObjects, List<DataObject> clientResults, Map<DataObject, USMException> errorMap, DataObject... extraDataObjects) throws USMException {
        long mt = computeTimestamp(serverDataObjects);
        createMissingUUIDs(serverDataObjects);
        mt = serverInterface.storeServerData(mt, serverDataObjects, incomplete);
        setTimestampAndCommitChanges(mt, serverDataObjects);
        DataObject[] dataToReturn = serverDataObjects.toArray();
        for (DataObject f : clientResults)
            f.setTimestamp(mt);
        for (DataObject f : extraDataObjects)
            f.setTimestamp(mt);
        return new SyncResult(incomplete, mt, dataToReturn, errorMap, DataObjectUtil.toArray(clientResults));
    }

    private static void createMissingUUIDs(Collection<DataObject> serverDataObjects) {
        for (DataObject o : serverDataObjects) {
            if (o.getUUID() == null) {
                throw new IllegalStateException("DataObject without UUID");
            }
        }
    }

    public static long updateTimestampAndCommitChanges(Collection<DataObject> serverDataObjects) {
        long timestamp = computeTimestamp(serverDataObjects);
        createMissingUUIDs(serverDataObjects);
        setTimestampAndCommitChanges(timestamp, serverDataObjects);
        return timestamp;
    }

    public static void setTimestampAndCommitChanges(long t, Collection<DataObject> serverDataObjects) {
        for (DataObject f : serverDataObjects) {
            f.setTimestamp(t);
            f.commitChanges();
        }
    }

    private static long computeTimestamp(Collection<DataObject> serverDataObjects) {
        long mt = 0;
        for (DataObject f : serverDataObjects)
            mt = Math.max(mt, f.getTimestamp());
        // If we have no objects and therefore no timestamp, use current system time as timestamp
        if (mt == 0)
            mt = System.currentTimeMillis();
        return mt;
    }

    public static void executeServerUpdates(SessionManagerImpl sessionManager, DataObjectSet serverObjects, List<DataObject> creations, List<DataObject> changes, List<DataObject> deletions, Map<DataObject, USMException> resultMap) {
        try {
            executeServerUpdatesInternal(sessionManager, serverObjects, creations, changes, deletions, resultMap, false, null);
        } catch (ConflictingChangeException e) {
            throw new IllegalStateException("Unexpected exception on update objects", e);
        }
    }

    public static void updateOnServer(SessionManagerImpl sessionManager, DataObjectSet serverObjects, List<DataObject> creations, List<DataObject> changes, List<DataObject> deletions, Map<DataObject, USMException> resultMap, List<DataObject> serverChanges) throws ConflictingChangeException {
        executeServerUpdatesInternal(sessionManager, serverObjects, creations, changes, deletions, resultMap, true, serverChanges);
    }

    protected static String getLatestMappedID(String originalID, Map<String, String> folderIdMapping) {
        String newMappedID = folderIdMapping.get(originalID);
        if (!folderIdMapping.containsKey(newMappedID))
            return newMappedID;
        return getLatestMappedID(newMappedID, folderIdMapping);
    }
    
    private static void executeServerUpdatesInternal(final SessionManagerImpl sessionManager, final DataObjectSet serverObjects, List<DataObject> creations, List<DataObject> changes, List<DataObject> deletions, Map<DataObject, USMException> resultMap, boolean throwConflictExceptions, final List<DataObject> serverChanges) throws ConflictingChangeException {
        final Map<String, String> folderIdMapping = new HashMap<String, String>();
        // perform changes on already existing objects
        performUpdates(serverObjects, changes, resultMap, throwConflictExceptions, new UpdateHandler() {

            @Override
            public void update(DataObject change, DataObject serverObject, boolean isFolder) throws USMException {
                if (folderIdMapping.containsKey(change.getOriginalParentFolderID()) || folderIdMapping.containsKey(change.getOriginalID())) {
                    DataObject serverObjectCopy = serverObject.createCopy(true);
                    DataObjectUtil.copyModifiedFields(serverObjectCopy, change);
                    if (folderIdMapping.containsKey(serverObjectCopy.getParentFolderID()))
                        serverObjectCopy.setParentFolderID(getLatestMappedID(serverObjectCopy.getParentFolderID(), folderIdMapping));
                    if (folderIdMapping.containsKey(serverObjectCopy.getID()))
                        serverObjectCopy.setID(getLatestMappedID(serverObjectCopy.getID(), folderIdMapping));

                    serverObjectCopy.getContentType().getTransferHandler().writeUpdatedDataObject(
                        serverObjectCopy,
                        serverObjectCopy.getTimestamp());
                    if (READ_SERVER_CHANGES_ON_MODIFICATION && isFolder) {
					    readServerChanges(sessionManager, serverChanges, serverObjectCopy, (Folder) serverObjectCopy, serverObjectCopy.getContentType());
                    }
                    DataObjectUtil.copyModifiedFields(change, serverObjectCopy);
                    DataObjectUtil.copyModifiedFields(serverObject, serverObjectCopy);
                    serverObject.commitChanges();
                } else {
                    change.getContentType().getTransferHandler().writeUpdatedDataObject(change, change.getTimestamp());
                    if (READ_SERVER_CHANGES_ON_MODIFICATION && isFolder) {
                        readServerChanges(sessionManager, serverChanges, change, (Folder) change, change.getContentType());
                    }
                    if (serverObject != null) {
                        DataObjectUtil.copyModifiedFields(serverObject, change);
                        serverObject.commitChanges();
                    }
                }
				if (isFolder && serverObject != null && change.isFieldModified(ID)) {
					//this means that we have renamed/moved a mail folder and the id is changed and we have to remap all cached data to the new id
                    Session session = serverObject.getSession();
                    String separator = getMailFolderIdSeparator(session, serverObject);
                    String originalID = change.getOriginalID();
                    String newID = serverObject.getID();
                    String originalIDPrefix = originalID + separator;
                    String newIDPrefix = newID + separator;

				    String newOriginalID = getLatestMappedID(originalID, folderIdMapping); //if the parent folder has also been renamed in the same update and we already have a new id mapped
                    if (newOriginalID != null)
                        originalID = newOriginalID;
                    folderIdMapping.put(originalID, newID); // old to new

                    session.remapCachedData(session.getShortFolderID(originalID, change.getUUID()), session.getShortFolderID(newID, change.getUUID()));
                    for (DataObject childFolder : serverObjects) {
                        String childFolderID = childFolder.getID();
                        boolean modified = false;
                        if (childFolderID.startsWith(originalIDPrefix)) {
                            String newChildFolderID = newIDPrefix + childFolderID.substring(originalIDPrefix.length());
                            folderIdMapping.put(childFolderID, newChildFolderID); // old to new
                            childFolder.setID(newChildFolderID);
                            session.remapCachedData(session.getShortFolderID(childFolderID, childFolder.getUUID()), session.getShortFolderID(newChildFolderID, childFolder.getUUID()));
                            modified = true;
                            }
                        String parentFolderID = childFolder.getParentFolderID();
                        if (parentFolderID.equals(originalID)) {
                            childFolder.setParentFolderID(newID);
                            modified = true;
                        } else if (parentFolderID.startsWith(originalIDPrefix)) {
                            String newParentFolderID = newIDPrefix + parentFolderID.substring(originalIDPrefix.length());
                            childFolder.setParentFolderID(newParentFolderID);
                            modified = true;
                        }
                        if (modified)
                            childFolder.commitChanges();
                        }
                    }
                }
        });
        // perform deletions of already existing objects
        performUpdates(serverObjects, deletions, resultMap, throwConflictExceptions, new UpdateHandler() {

            @Override
            public void update(DataObject deletion, DataObject serverObject, boolean isFolder) throws USMException {
                boolean executeDeletion = true;
                if (isFolder && serverObject != null) {
                    Folder f = serverObject.getParentFolder();
                    executeDeletion = f == null || serverObjects.contains(f);
                }
                if (executeDeletion)
                    deletion.getContentType().getTransferHandler().writeDeletedDataObject(deletion);
                if (serverObject != null)
                    serverObjects.remove(serverObject);
            }
        });
        // perform creations of new objects
        performUpdates(null, creations, resultMap, throwConflictExceptions, new UpdateHandler() {

            private DataObjectSet _newestCachedData = null;

            @Override
            public void update(DataObject creation, DataObject serverObject, boolean isFolder) throws USMException {
                Folder f = creation.getParentFolder();
                if (f != null)
                    creation.setParentFolder(f);
                ContentType contentType = creation.getContentType();
                contentType.getTransferHandler().writeNewDataObject(creation);
                if(serverObjects.contains(creation.getID()))
				    throw new USMException(USMSessionManagementErrorCodes.DUPLICATE_FOLDER_CREATION, "Folder already exists");
                DataObject creationCopy = creation.createCopy(true);
                if (READ_SERVER_CHANGES_ON_CREATION) {
                    readServerChanges(sessionManager, serverChanges, creation, f, contentType);
                }
                serverObjects.add(creationCopy);
               if (creation.getUUID() == null) {
					_newestCachedData = sessionManager.insertStoredUUID(creation, null, _newestCachedData);
					if (creation.getUUID() == null)
						creation.setUUID(UUID.randomUUID());
				} 
            }
        });
    }

  
    protected static String getMailFolderIdSeparator(Session session, DataObject serverObject) {
        String separator = "/"; // default ?
        String folderId = serverObject.getID();
        if (folderId.startsWith("default")) { // is a mail folder
            String part = folderId.substring("default".length());
            // search longest account id that matches
            for (Map.Entry<String, String> entry : session.getMailFolderIdSeparators().entrySet()) {
                String accountId = entry.getKey();
                if (accountId.length() >= separator.length() && part.startsWith(accountId))
                    separator = entry.getValue();
            }
        }
        return separator;
    }

    protected static void readServerChanges(final SessionManagerImpl sessionManager,
			final List<DataObject> serverChanges, DataObject creation, Folder f, ContentType contentType) {
		if (mayHaveRightsToReadElements(f)) {
			try {
				DataObject creationCopy2 = creation.createCopy(true);
				creationCopy2.commitChanges();
				contentType.getTransferHandler().readDataObject(creationCopy2,
						creationCopy2.getSession().getFieldFilter(contentType));
				if (creationCopy2.isModified() && serverChanges != null)
					serverChanges.add(creationCopy2);
			} catch (USMException e) {
				sessionManager.getJournal().debug(
						creation.getSession() + " Couldn't read implicit server changes for client creation", e);
			}
		}
	}

    /**
	 * Checks given folder if it may have read rights for the current user.
	 * - If f is null, it returns false
	 * - if the field "own_rights" is not set,. it returns true (since the rights have not been read from the OX server for that folder)
	 * - otherwise it returns true, if any read rights exist (for own or for all objects), i.e. if bits 7-13 are not all 0.
     * 
     * @param f
     * @return
     */
    private static boolean mayHaveRightsToReadElements(Folder f) {
        if (f == null)
            return false;
        Object v = f.getFieldContent("own_rights");
        if (!(v instanceof Number))
            return true;
        int rights = ((Number) v).intValue();
        return (rights & 0x3F80) != 0;
    }

    private static void performUpdates(DataObjectSet serverObjects, List<DataObject> updates, Map<DataObject, USMException> resultMap, boolean throwConflictExceptions, UpdateHandler handler) throws ConflictingChangeException {
        if (updates == null || updates.isEmpty())
            return;
        if (performFolderUpdates(serverObjects, updates, resultMap, throwConflictExceptions, handler))
            return;
        for (DataObject update : updates)
            performOneUpdate(serverObjects, resultMap, throwConflictExceptions, handler, update, false);
    }

    private static boolean performFolderUpdates(DataObjectSet serverObjects, List<DataObject> updates, Map<DataObject, USMException> resultMap, boolean throwConflictExceptions, UpdateHandler handler) throws ConflictingChangeException {
        if (updates.get(0).getContentType().getCode() != DefaultContentTypes.FOLDER_CODE)
            return false;
        Folder[] folders = FolderHierarchyComparator.convertToSortedFoldersArray(updates);
        if (folders == null)
            return false;
        for (Folder f : folders)
            performOneUpdate(serverObjects, resultMap, throwConflictExceptions, handler, f, true);
        return true;
    }

    private static void performOneUpdate(DataObjectSet serverObjects, Map<DataObject, USMException> resultMap, boolean throwConflictExceptions, UpdateHandler handler, DataObject update, boolean isFolderUpdate) throws ConflictingChangeException {
        try {
            DataObject serverObject = null;
            if (serverObjects != null) {
                serverObject = serverObjects.get(update.getID());
                if (serverObject != null)
                    update.setTimestamp(serverObject.getTimestamp());
            }
            handler.update(update, serverObject, isFolderUpdate);
        } catch (ConflictingChangeException e) {
            if (throwConflictExceptions)
                throw e;
            resultMap.put(update, e);
        } catch (USMException e) {
            resultMap.put(update, e);
        }
    }
}
