/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.sync;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import org.apache.commons.logging.Log;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.DataObjectFilter;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.assets.ChangeState;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.impl.SessionManagerImpl;

public abstract class BaseContentSyncer {

    protected final SessionManagerImpl _sessionManager;

    protected final int _retryCount;

    protected CachedOXData _oxDataCache;

    protected final Log _journal;

    public BaseContentSyncer(SessionManagerImpl sessionManager, int retryCount) {
        _sessionManager = sessionManager;
        _retryCount = retryCount;
        _journal = _sessionManager.getJournal();
    }

    protected boolean shouldBeAddedToResult(int resultLimit, int clientsResultSize, DataObjectFilter filter, ChangeState action, DataObject object, Set<Object> objectGroups) {
        Object objectGroupOwner = object.getContentType().getObjectGroupOwner(object);
        if (objectGroups.contains(objectGroupOwner))
            return true;
        if (resultLimit != Session.NO_LIMIT && clientsResultSize >= resultLimit)
            return false;
        if (filter != null && !filter.addOperationToResult(action, object))
            return false;
        objectGroups.add(objectGroupOwner);
        return true;
    }

    protected Collection<? extends DataObject> sortElements(Collection<? extends DataObject> data, Comparator<? super DataObject> sorter) {
        if (sorter == null || data.size() <= 1)
            return data;
        List<? extends DataObject> list = (data instanceof List) ? (List<? extends DataObject>) data : (data instanceof DataObjectSet) ? ((DataObjectSet) data).toList() : new ArrayList<DataObject>(
            data);
        Collections.sort(list, sorter);
        return list;
    }

    protected DataObjectSet getCurrentOrCachedServerData(ContentSyncerStorage storage, boolean allowCache) throws USMException {
        if (allowCache) {
            _oxDataCache = _sessionManager.getOXDataCache().getCachedData(storage.getCacheID());
            if (_oxDataCache != null) {
                if (_journal.isDebugEnabled())
                    _journal.debug(storage.getCacheID() + " Using cached OX server data for sync");
                return new DataObjectSet(_oxDataCache.getData());
            }
        }
        DataObjectSet serverDataObjects = new DataObjectSet();
        boolean isComplete = storage.getCurrentServerData(serverDataObjects);
        if (allowCache && isComplete) {
            _oxDataCache = _sessionManager.getOXDataCache().storeCachedData(storage.getCacheID(), serverDataObjects.toArray());
        } else {
            _oxDataCache = null;
            _sessionManager.getOXDataCache().removeCachedData(storage.getCacheID());
        }
        return serverDataObjects;
    }

    protected void setParentFolderOwnerIDField(DataObject[] dataObjects) throws StorageAccessException, USMStorageException {
        Folder parentFolder = null;
        boolean searchedCachedFolders = false;
        for (DataObject dataObject : dataObjects) {
            Folder folder = dataObject.getParentFolder();
            if (folder != null) {
                parentFolder = folder;
                dataObject.setParentFolderOwnerID(folder.getOwnerID());
            } else {
                if (!searchedCachedFolders && parentFolder == null)
                    parentFolder = findParentFolder(dataObject);
                if (parentFolder != null) {
                    dataObject.setParentFolderOwnerID(parentFolder.getOwnerID());
                    dataObject.setParentFolder(parentFolder);
                }
            }
        }
    }

    private Folder findParentFolder(DataObject object) throws StorageAccessException, USMStorageException {
        String parentFolderID = object.getOriginalParentFolderID();
        if (parentFolderID == null)
            parentFolderID = object.getParentFolderID();
        if (parentFolderID != null) {
            Folder[] cachedFolders = object.getSession().getCachedFolders();
            if (cachedFolders != null) {
                for (Folder folder2 : cachedFolders) {
                    if (folder2.getID().equals(parentFolderID))
                        return folder2;
                }
            }
        }
        return null;
    }
}
