/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.dataobject;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.UUID;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.folder.FolderConstants;
import com.openexchange.usm.api.datatypes.DataType;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;

public class DataObjectUtil {

    public static DataObject findDataObject(String objectID, DataObject... objects) {
        if (objectID != null && objects != null) {
            for (DataObject o : objects) {
                if (objectID.equals(o.getID()))
                    return o;
            }
        }
        return null;
    }

    public static DataObject findDataObject(String objectID, Collection<DataObject> objects) {
        if (objectID != null && objects != null) {
            for (DataObject o : objects) {
                if (objectID.equals(o.getID()))
                    return o;
            }
        }
        return null;
    }

    public static DataObject findDataObject(UUID uuid, DataObject... objects) {
        if (uuid != null && objects != null) {
            for (DataObject o : objects) {
                if (uuid.equals(o.getUUID()))
                    return o;
            }
        }
        return null;
    }

    public static DataObject findDataObject(UUID uuid, Collection<DataObject> objects) {
        if (uuid != null && objects != null) {
            for (DataObject o : objects) {
                if (uuid.equals(o.getUUID()))
                    return o;
            }
        }
        return null;
    }

    public static void modify(DataObject object, DataObject modification) {
        modify(object, modification, false);
    }

    public static void modify(DataObject object, DataObject modification, boolean ignoreNullFields) {
        ContentType type = object.getContentType();
        int len = type.getFields().length;
        // The following 3 calls may not all be necessary, depending on implementation (some or all of the information may actually be
        // stored in fields), but a duplicate copy doesn't hurt
//        object.setID(modification.getID());
//        object.setParentFolderID(modification.getParentFolderID());
        object.setTimestamp(modification.getTimestamp());
        for (int i = 0; i < len; i++) {
            Object fieldContent = modification.getFieldContent(i);
            if(!(ignoreNullFields && fieldContent == null))
                object.setFieldContent(i, fieldContent);
        }
    }

    public static DataObject copyAndModify(DataObject source, DataObject modification, boolean linkUUIDs) {
      return copyAndModify(source, modification, linkUUIDs, false);
    }
    
    public static DataObject copyAndModify(DataObject source, DataObject modification, boolean linkUUIDs, boolean ignoreNullFields) {
        if (!source.getContentType().getID().equals(modification.getContentType().getID()))
            throw new IllegalArgumentException("ContentType mismatch");
        DataObject result = source.createCopy(linkUUIDs);
        result.commitChanges();
        modify(result, modification, ignoreNullFields);
        return result;
    }

    public static DataObject copyModifications(DataObject object, DataObject modification) {
        if (!object.getContentType().getID().equals(modification.getContentType().getID()))
            throw new IllegalArgumentException("ContentType mismatch");
        // The following 3 calls may not all be necessary, depending on implementation (some or all of the information may actually be
        // stored in fields), but a duplicate copy doesn't hurt
        object.setID(modification.getID());
        object.setParentFolderID(modification.getParentFolderID());
        return copyModifiedFields(object, modification);
    }

    public static DataObject copyModifiedFields(DataObject object, DataObject modification) {
        object.setTimestamp(modification.getTimestamp());
        int len = object.getContentType().getFields().length;
        for (int i = 0; i < len; i++) {
            if (modification.isFieldModified(i))
                object.setFieldContent(i, modification.getFieldContent(i));
        }
        return object;
    }

    public static boolean isFieldEqual(DataObject o1, DataObject o2, String field) {
        if (!o1.getContentType().getID().equals(o2.getContentType().getID()))
            throw new IllegalArgumentException("ContentType mismatch");
        int index = o1.getFieldIndex(field);
        DataType<?> dataType = o1.getContentType().getFields()[index].getFieldType();
        return dataType.isEqual(o1.getFieldContent(index), o2.getFieldContent(index));
    }

    public static boolean isEqual(Object v1, Object v2) {
        return (v1 == null) ? v2 == null : v1.equals(v2);
    }

    public static List<DataObject> toList(DataObject[] array) {
        List<DataObject> result = new ArrayList<DataObject>(array.length);
        for (DataObject o : array)
            result.add(o);
        return result;
    }

    public static DataObject[] toArray(Collection<DataObject> collection) {
        return collection.toArray(new DataObject[collection.size()]);
    }

    public static long getLongValue(DataObject o, String fieldName) {
        Object f = o.getFieldContent(fieldName);
        return (f instanceof Number) ? ((Number) f).longValue() : 0L;
    }

    public static int getIntValue(DataObject o, String fieldName) {
        Object f = o.getFieldContent(fieldName);
        return (f instanceof Number) ? ((Number) f).intValue() : 0;
    }

    public static DataObject findMailFolder(String parentFolderId, String folderTitle, Iterable<DataObject> objects) {
        if (parentFolderId != null && folderTitle != null) {
            for (DataObject o : objects) {
                if (o instanceof Folder) {
                    Folder f = (Folder) o;
                    try {
                        if (DefaultContentTypes.MAIL_ID.equals(f.getElementsContentTypeID()) && parentFolderId.equals(o.getParentFolderID()) && folderTitle.equals(o.getFieldContent(FolderConstants.TITLE)))
                            return o;
                    } catch (Exception ignored) {
                        // skip this object
                    }
                }
            }
        }
        return null;
    }
}
