/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.mimemail;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Enumeration;
import java.util.Properties;
import javax.activation.DataHandler;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimePart;
import javax.mail.util.ByteArrayDataSource;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.connector.commands.CommandConstants;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.util.Toolkit;

/**
 * This class converts the JSON structure with information about the mail to the mime mail. 
 * The JAVA Mail API is used for this aim.
 * @author ibr
 *
 */
public class MimeMailBuilder extends AbstractMimeMailBuilder {

	private static final String _MULTIPART_MIXED = "multipart/mixed";
	
	public String convertJSONObjectToMimeMail(JSONObject data) throws USMJSONAPIException {
		if (data.has(SMIME_BODY_DATA)) {
			if (data.has(BODY))
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_UNSUPPORTED_BODY_FIELDS,
						ResponseStatusCode.BAD_REQUEST, "Message contains smime body field and other body fields!");
			return buildSmimeMailFromJSONObject(data);
		}
        buildMimeMailFromJSONObject(data);
        return writeToString();
	}

	private String buildSmimeMailFromJSONObject(JSONObject data) throws USMJSONAPIException {
		JSONObject headers = data.optJSONObject(HEADERS);
		StringBuilder sb = new StringBuilder(1000);
		try {
			Object contentType;
			contentType = (headers != null && headers.has(CONTENT_TYPE)) ? headers.get(CONTENT_TYPE) : null;
			String contentTypeOfPart = buildContentTypeAsString(contentType);
			//add headers to message
			MimePart part = new MimeBodyPart();
			appendHeadersToPart(part, headers, contentTypeOfPart);
			for (Enumeration<?> e = part.getAllHeaderLines(); e.hasMoreElements();) {
				sb.append((String) e.nextElement());
				sb.append("\r\n");
			}
			sb.append(new String(Toolkit.decodeBase64(data.getString(SMIME_BODY_DATA))));
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.SMIMEMAIL_JSONEXCEPTION_ERROR,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "JSONException: " + e.getMessage());
		} catch (MessagingException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.SMIMEMAIL_JAVAPI_MESSAGING_ERROR,
					ResponseStatusCode.INTERNAL_ERROR, "error by building of body part: " + e.getMessage());
		}
		return sb.toString();
	}

	@Override
    protected void buildPart(MimePart part, JSONObject json) throws USMJSONAPIException {
		try {
            JSONObject headers = json.optJSONObject(HEADERS);
            Object contentType = (headers != null && headers.has(CONTENT_TYPE)) ? headers.get(CONTENT_TYPE) : _DEFAULT_CONTENT_TYPE;
            String contentTypeOfPart = buildContentTypeAsString(contentType);
			if (contentTypeOfPart.startsWith(_MULTIPART_MIXED)) {
				//add headers to message
				appendHeadersToPart(part, headers, _MULTIPART_MIXED);
				// set multi part content
				part.setContent(buildMimePartForMultipart(json, "mixed"), _MULTIPART_MIXED);
			} else if (contentTypeOfPart.startsWith(MULTIPART)) {
				//add headers to message
				appendHeadersToPart(part, headers, contentTypeOfPart);
				// set multi part content
				String subtype = contentTypeOfPart.replaceFirst(MULTIPART, "");
				part.setContent(buildMimePartForMultipart(json, subtype), contentTypeOfPart);

			} else {
				JSONObject body = json.has(BODY) ? json.getJSONObject(BODY) : null;
				if (contentTypeOfPart.startsWith(CONTENTTYPE_TEXT_PREFIX)) {
					try {
						//add headers
						String encoding = getValue(headers, CONTENT_TRANSFER_ENCODING, 0, "quoted-printable");
						setHeaderConditional(part, CONTENT_TRANSFER_ENCODING_UPPER, encoding);
						appendHeadersToPart(part, headers, contentTypeOfPart);
						//build text part
						String content = (body != null && body.has(CommandConstants.DATA_KEY)) ? body.getString(CommandConstants.DATA_KEY) : "";
						buildTextPart(part, contentTypeOfPart, content);
					} catch (UnsupportedEncodingException e) {
						throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_UNSUPPORTED_ENCODING_ERROR_3,
								ResponseStatusCode.INTERNAL_ERROR, "unsupported encoding of text content: "
										+ e.getMessage());

					}
				} else if (body != null && body.has(CommandConstants.DATA_KEY)) {
					String mailcontentTypeAttachment = extractPlainContentType(contentTypeOfPart).trim();
					//add headers
					appendHeadersToPart(part, headers, contentTypeOfPart);
					// set content
					String content = body.getString(CommandConstants.DATA_KEY);
					String encoding = getValue(headers, CONTENT_TRANSFER_ENCODING, 0, null);
					if (CommandConstants.BASE64.equals(encoding)) {
						setHeaderConditional(part, CONTENT_TRANSFER_ENCODING_UPPER, encoding);
						part.setDataHandler(new DataHandler(new ByteArrayDataSource(Toolkit.decodeBase64(content),
								mailcontentTypeAttachment)));
					} else {
						part.setContent(content, mailcontentTypeAttachment);
					}
					part.setHeader(CONTENT_TYPE_UPPER, contentTypeOfPart);
				} else if (contentTypeOfPart.startsWith(CONTENTTYPE_RFC822)) {
					appendHeadersToPart(part, headers, contentTypeOfPart);
					USMMimeMessage nestedPart = new USMMimeMessage(Session.getInstance(new Properties()));
					buildPart(nestedPart, body);
					part.setContent(nestedPart, CONTENTTYPE_RFC822);
				}
			}
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_JSONEXCEPTION_ERROR,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "JSONException: " + e.getMessage());

		} catch (MessagingException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_JAVAPI_MESSAGING_ERROR,
					ResponseStatusCode.INTERNAL_ERROR, "error by building of body part: " + e.getMessage());
		}

	}

	private static void buildTextPart(MimePart part, String contentTypeOfPart, String decodedContent) throws MessagingException, UnsupportedEncodingException {
		String charset = extractEncodingCharset(contentTypeOfPart);
		// set content
		byte[] contentBytes;
		//decode if Content-Transfer-Encoding = Base64
		if (CommandConstants.BASE64.equals(part.getEncoding()))
			contentBytes = Toolkit.decodeBase64(decodedContent);
		else
			contentBytes = decodedContent.getBytes(charset);
	    part.setDataHandler(new DataHandler(new ByteArrayDataSource(contentBytes, contentTypeOfPart)));
	    part.setHeader("Content-Type", contentTypeOfPart);
	}

    private String writeToString() throws USMJSONAPIException {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();

        try {
            _message.writeTo(outputStream);
        } catch (IOException e) {
            throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_WRITE_OUT_ERROR,
                    ResponseStatusCode.INTERNAL_ERROR, "error by output of mime mail: " + e.getMessage(), e);

        } catch (MessagingException e) {
            throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_WRITE_OUT_ERROR,
                    ResponseStatusCode.INTERNAL_ERROR, "messaging error by output of mime mail: " + e.getMessage(), e);
        } finally {
            try {
                outputStream.close();
            } catch (IOException e) {
                throw new USMJSONAPIException(
                    ConnectorBundleErrorCodes.MIMEMAIL_IO_ERROR,
                    ResponseStatusCode.INTERNAL_ERROR,
                    "IO error by output of mime mail: " + e.getMessage(),
                    e);
            }
        }
        return outputStream.toString();
    }
}
