/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.json.streaming;

import java.text.ParseException;
import com.openexchange.usm.api.datatypes.PIMAttachment;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.response.ResponseStatusCode;

/**
 * {@link ServerTempId}
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class ServerTempId {

    private final static boolean _ENCODE_UNCOMMON_CHARACTERS = false;

    private final static char _DELIMITER = '_';

    private final static String _HEX_CHARS = "0123456789abcdef";

    /**
     * Creates a new {@link ServerTempId} from a PIM Object and one of its PIM-Attachments.
     * 
     * @param o
     * @param att
     * @return new ServerTempId for the given PIM attachment
     */
    public static ServerTempId forPIMAttachment(DataObject o, PIMAttachment att) {
        return new ServerTempId(ServerTempIdType.PIM, o.getParentFolderID(), o.getID(), Integer.toString(att.getOxId()), att.getFileSize());
    }

    /**
     * Creates a new {@link ServerTempId} from an Email object and the attachment ID of one of its mail attachments
     * 
     * @param mailObject
     * @param attachmentId
     * @param attachmentSize
     * @return new ServerTempId for the given mail attachment
     */
    public static ServerTempId forMailAttachment(DataObject mailObject, String attachmentId, long attachmentSize) {
        return new ServerTempId(ServerTempIdType.MAIL, mailObject.getParentFolderID(), mailObject.getID(), attachmentId, attachmentSize);
    }

    /**
     * Creates a new {@link ServerTempId} for the image of a Contactl object
     * 
     * @param contact
     * @return new ServerTempId for the image of the given contact
     */
    public static ServerTempId forContactImage(DataObject contact) {
        return new ServerTempId(ServerTempIdType.IMAGE, contact.getParentFolderID(), contact.getID(), "1", 1);
    }

    /**
     * Parses a tempid and extracts the data that was stored by the server. If the tempid is not of the correct format, throws an
     * USMJSONAPIException.
     * 
     * @param s String as generated from ServerTempId.toString()
     * @return ServerTempId with parsed information
     * @throws USMJSONAPIException if format is not correct or s is null
     */
    public static ServerTempId fromString(String s) throws USMJSONAPIException {
        if (s == null)
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.TEMPID_NOT_SET,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "tempid was not set");
        int split = s.indexOf(_DELIMITER);
        if (split >= 0) {
            try {
                ServerTempIdType type = ServerTempIdType.valueOf(s.substring(0, split));
                String id = s.substring(split + 1);
                ServerTempId result = null;
                if (id.indexOf("__") < 0) {
                    result = parseSimpleId(type, id);
                } else if (type == ServerTempIdType.MAIL) {
                    result = parseComplexId(id);
                }
                if (result != null)
                    return result;
            } catch (ParseException e) {
                // fall through to error
            } catch (NumberFormatException e) {
                // fall through to error
            }
        }
        throw new USMJSONAPIException(
            ConnectorBundleErrorCodes.INVALID_TEMPID,
            ResponseStatusCode.WRONG_MISSING_PARAMETERS,
            "Illegal server tempid '" + s + '\'');
    }

    private static ServerTempId parseComplexId(String id) throws ParseException {
        StringBuilder folderId = new StringBuilder();
        int pos = parseObjectId(id, 0, folderId);
        StringBuilder objectId = new StringBuilder();
        pos = parseObjectId(id, pos, objectId);
        StringBuilder attachmentId = new StringBuilder();
        pos = parseObjectId(id, pos, attachmentId);
        if (pos >= id.length())
            return null;
        long size = Long.parseLong(id.substring(pos));
        return new ServerTempId(ServerTempIdType.MAIL, folderId.toString(), objectId.toString(), attachmentId.toString(), size);
    }

    private static ServerTempId parseSimpleId(ServerTempIdType type, String id) {
        int p1 = id.indexOf(_DELIMITER);
        if (p1 <= 0)
            return null;
        int p2 = id.indexOf(_DELIMITER, p1 + 1);
        if (p2 <= p1)
            return null;
        int p3 = id.indexOf(_DELIMITER, p2 + 1);
        if (p3 <= p2)
            return null;
        String attId = id.substring(p2 + 1, p3);
        switch (type) {
        case IMAGE: // Verify that attachmentId is 1 (not used by contact images)
            if (!"1".equals(attId))
                return null;
            break;
        case PIM: // Verify that attachmentId is an integer value
            Integer.parseInt(attId);
            break;
        case MAIL:
            break;
        default:
            break;
        }
        long size = Long.parseLong(id.substring(p3 + 1));
        return new ServerTempId(type, id.substring(0, p1), id.substring(p1 + 1, p2), attId, size);
    }

    private static int parseObjectId(String id, int start, StringBuilder part) throws ParseException {
        for (int i = start; i < id.length(); i++) {
            char c = id.charAt(i);
            if (c == _DELIMITER) {
                int end = i + 1;
                if (end >= id.length() || id.charAt(end) != _DELIMITER)
                    return end;
                if (_ENCODE_UNCOMMON_CHARACTERS) {
                    if (end >= id.length() - 6)
                        throw new ParseException("Can not parse part of '" + id + '\'', start);
                    part.append((char) Integer.parseInt(id.substring(end + 1, end + 5), 16));
                    i += 5;
                } else {
                    part.append(_DELIMITER);
                    i++;
                }
            } else {
                part.append(c);
            }
        }
        return id.length();
    }

    private static void addObjectIdToTempId(StringBuilder sb, String id) {
        for (int i = 0; i < id.length(); i++) {
            char c = id.charAt(i);
            if (_ENCODE_UNCOMMON_CHARACTERS) {
                if (isSimpleChar(c)) {
                    sb.append(c);
                } else {
                    // escape complex characters with __uuuu where uuuu is the unicode in hex
                    sb.append(_DELIMITER).append(_DELIMITER);
                    sb.append(_HEX_CHARS.charAt(c >> 12));
                    sb.append(_HEX_CHARS.charAt((c >> 8) & 0xF));
                    sb.append(_HEX_CHARS.charAt((c >> 4) & 0xF));
                    sb.append(_HEX_CHARS.charAt(c & 0xF));
                }
            } else {
                if (c == _DELIMITER)
                    sb.append(_DELIMITER);
                sb.append(c);
            }
        }
    }

    private static boolean isSimpleChar(char c) {
        return (c >= '0' && c <= '9') || (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z');
    }

    private final ServerTempIdType _type;

    private final String _folderId;

    private final String _objectId;

    private final String _attachmentId;

    private final long _size;

    private ServerTempId(ServerTempIdType type, String folderId, String objectId, String attachmentId, long size) {
        _type = type;
        _folderId = folderId;
        _objectId = objectId;
        _attachmentId = attachmentId;
        _size = size;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(_type);
        sb.append(_DELIMITER);
        addObjectIdToTempId(sb, _folderId);
        sb.append(_DELIMITER);
        addObjectIdToTempId(sb, _objectId);
        sb.append(_DELIMITER);
        addObjectIdToTempId(sb, _attachmentId);
        sb.append(_DELIMITER);
        sb.append(_size);
        return sb.toString();
    }

    public ServerTempIdType getType() {
        return _type;
    }

    public String getFolderId() {
        return _folderId;
    }

    public String getObjectId() {
        return _objectId;
    }

    public String getAttachmentId() {
        return _attachmentId;
    }

    public long getSize() {
        return _size;
    }
}
