/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.json.streaming;

import static com.openexchange.usm.connector.commands.CommandConstants.DATA_KEY;
import static com.openexchange.usm.connector.commands.CommandConstants.EXACT_SIZE;
import static com.openexchange.usm.connector.commands.CommandConstants.ID;
import static com.openexchange.usm.connector.commands.CommandConstants.TEMPID;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.CommonConstants;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.util.JSONToolkit;

/**
 * {@link MailAttachmentStreamingUtil} Helper class that parses a structured mail JSONObject as returned from OX, and replaces all "data"
 * fields with a new value with a "tempid" field that can later be used to download the attachment via the "downloadData" command.
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public final class MailAttachmentStreamingUtil {

    public static void replaceEmptyDataWithTempId(DataObject mail, JSONObject structuredMail, boolean supportsAttachmentStreaming) {
        if (structuredMail.has(ID) && structuredMail.has(DATA_KEY) && structuredMail.has(EXACT_SIZE) && structuredMail.isNull(DATA_KEY) && !structuredMail.has(TEMPID)) {
            try {
                if (supportsAttachmentStreaming) {
                    String attachmentId = structuredMail.optString(CommonConstants.ID);
                    long exactSize = structuredMail.optLong(EXACT_SIZE);
                    structuredMail.put(TEMPID, ServerTempId.forMailAttachment(mail, attachmentId, exactSize));
                    structuredMail.remove(DATA_KEY);
                    structuredMail.remove(EXACT_SIZE);
                } else {
                    structuredMail.put(DATA_KEY, "");
                }
            } catch (JSONException ignored) {
                // can not happen
            }
        }
        for (String key : JSONToolkit.keys(structuredMail)) {
            Object o = structuredMail.opt(key);
            if (o instanceof JSONObject)
                replaceEmptyDataWithTempId(mail, (JSONObject) o, supportsAttachmentStreaming);
            else if (o instanceof JSONArray)
                checkMailSubArray(mail, (JSONArray) o, supportsAttachmentStreaming);
        }
    }

    private static void checkMailSubArray(DataObject mail, JSONArray array, boolean supportsAttachmentStreaming) {
        for (int i = 0; i < array.length(); i++) {
            Object o = array.opt(i);
            if (o instanceof JSONObject)
                replaceEmptyDataWithTempId(mail, (JSONObject) o, supportsAttachmentStreaming);
            else if (o instanceof JSONArray)
                checkMailSubArray(mail, (JSONArray) o, supportsAttachmentStreaming);
        }
    }

    private MailAttachmentStreamingUtil() {
    }
}
