/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.json.response;

import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.exceptions.USMJSONException;
import com.openexchange.usm.json.USMJSONAPIException;

/**
 * The JSONObject to be returned in the Response of any USM-JSON-Command.
 * @author ldo
 *
 */
public class ResponseObject extends JSONObject {

	public static final String DATA = "data";
	public static final String STATUS = "status";
	public static final String ERROR_CODE = "errorCode";
	public static final String ERROR_MESSAGE = "errorMessage";
	public static final String ERROR_DETAILS = "errorDetails";
	public static final String ERROR_STATUS = "errorStatus";
	public static final String OX_ERROR = "oxError";

	private final ResponseStatusCode _status;
	private final Object _data;
	private final int _errorCode;
	private final String _errorMessage;
	private final JSONObject _errorDetails;
	private final JSONObject _oxError;
	private final USMJSONException _exception;

	public ResponseObject(ResponseStatusCode status) {
		this(status, null, 0, null, null, null, null);
	}

	public ResponseObject(ResponseStatusCode status, Object data) {
		this(status, data, 0, null, null, null, null);
	}

	public ResponseObject(ResponseStatusCode status, int errorCode, String errorMessage, JSONObject errorDetails,
			JSONObject oxError) {
		this(status, null, errorCode, errorMessage, errorDetails, oxError, null);
	}

	public ResponseObject(ResponseStatusCode status, USMJSONException error) {
		this(status, null, error.getErrorCode(), error.getErrorMessage(), error.getErrorDetailsForJSONResponse(), error
				.getOxErrorForJSONResponse(), error);
	}

	public ResponseObject(ResponseStatusCode status, Object data, USMJSONException error) {
		this(status, data, error.getErrorCode(), error.getErrorMessage(), error.getErrorDetailsForJSONResponse(), error
				.getOxErrorForJSONResponse(), error);
	}

	public ResponseObject(USMJSONAPIException error) {
		this(error.getResponseStatusCode(), null, error.getErrorCode(), error.getErrorMessage(), error
				.getErrorDetailsForJSONResponse(), error.getOxErrorForJSONResponse(), error);
	}

	public ResponseObject(ResponseStatusCode status, Object data, int errorCode, String errorMessage,
			JSONObject errorDetails, JSONObject oxError, USMJSONException cause) {
		super();
		_status = status;
		_data = data;
		_errorCode = errorCode;
		_errorMessage = errorMessage;
		_errorDetails = errorDetails;
		_oxError = oxError;
		_exception = cause;
		try {
			if (_status != null)
				this.put(STATUS, _status.getStatusCode());
			if (_data != null)
				this.put(DATA, _data);
			if (_errorCode != 0)
				this.put(ERROR_CODE, _errorCode);
			if (_errorMessage != null)
				this.put(ERROR_MESSAGE, _errorMessage);
			if (_errorDetails != null)
				this.put(ERROR_DETAILS, _errorDetails);
			if (_oxError != null)
				this.put(OX_ERROR, _oxError);
		} catch (JSONException ignored) {
			// Ignored since it can never happen with our fixed keys
		}
	}

	public ResponseStatusCode getStatusCode() {
		return _status;
	}

	public Object getData() {
		return _data;
	}

	public int getErrorCode() {
		return _errorCode;
	}

	public String getErrorMessage() {
		return _errorMessage;
	}

	public JSONObject getErrorDetails() {
		return _errorDetails;
	}

	public JSONObject getOxError() {
		return _oxError;
	}

	public USMJSONException getCause() {
		return _exception;
	}
}
