/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.json;

import java.util.ArrayList;
import java.util.List;

import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.util.Version;

/**
 * Contains version information as static String, contains version history as comments. Methods are provided to store/retrieve the version
 * used by a specific client from the USM session and to test for supported capabilities
 * 
 * @author afe
 */
public final class USMJSONVersion {

    /**
     * USM JSON versions
     */
    public static final String VERSION = "7.10.6-11";

    public static final String BUILD_NUMBER = "e41171e3c52e1154110c99fab93507a023973cf1- 7477";

    public static final String _CLIENT_USM_JSON_VERSION_KEY = "json.Prot.Version";

    private static final USMJSONVersion _SUPPORTS_FILTERED_OUT_LARGE_OBJECTS_STATUS_CODE = new USMJSONVersion("1.6.10");

    public static final USMJSONVersion SUPPORTS_ATTACHMENT_STREAMING = new USMJSONVersion("2.0.0");

    public static final USMJSONVersion SUPPORTS_CONTACT_IMAGE_STREAMING = new USMJSONVersion("2.2.0");
    
    public static final USMJSONVersion SUPPORTS_STANDARD_FOLDER_SWAPPING = new USMJSONVersion("2.3.0");
    
    public static final USMJSONVersion SUPPORTS_CONTACT_IMAGE_STREAMING_FALSE_33754 = new USMJSONVersion("2.1.0");

    private static final USMJSONVersion _CURRENT_VERSION = _SUPPORTS_FILTERED_OUT_LARGE_OBJECTS_STATUS_CODE;

    private static final USMJSONVersion _NEWEST_VERSION = SUPPORTS_STANDARD_FOLDER_SWAPPING;

    private static final List<USMJSONVersion> _supportedVersions = new ArrayList<USMJSONVersion>();

    private static List<USMJSONVersion> getSupportedVersions() {
        if (_supportedVersions.isEmpty()) {
            synchronized (_supportedVersions) {
                if (_supportedVersions.isEmpty()) {
                    _supportedVersions.add(SUPPORTS_STANDARD_FOLDER_SWAPPING);
                    _supportedVersions.add(SUPPORTS_CONTACT_IMAGE_STREAMING);
                    _supportedVersions.add(SUPPORTS_CONTACT_IMAGE_STREAMING_FALSE_33754);
                    _supportedVersions.add(SUPPORTS_ATTACHMENT_STREAMING);
                    _supportedVersions.add(_SUPPORTS_FILTERED_OUT_LARGE_OBJECTS_STATUS_CODE);
                }
            }
        }
        return _supportedVersions;
    }

    public static Version getCurrentVersion() {
        return _CURRENT_VERSION.getVersion();
    }

    public static Version getNewestVersion() {
        return _NEWEST_VERSION.getVersion();
    }

    public static Version getMatchingVersion(Version version) {
        if (version == null)
            return getCurrentVersion();
        for (USMJSONVersion v : getSupportedVersions()) {
            if (v.matches(version))
                return v.getVersion();
        }
        for (USMJSONVersion v : getSupportedVersions()) {
            if (v.supports(version))
                return v.getVersion();
        }
        return null;
    }

    public static void storeUsedVersion(Session session, Version version) throws USMJSONAPIException {
        if (version == null)
            version = getCurrentVersion();
        try {
            session.setPersistentField(_CLIENT_USM_JSON_VERSION_KEY, version.toString());
        } catch (StorageAccessException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.DB_ERROR_WHILE_STORING_PROTOCOL_VERSION, e);
        } catch (USMStorageException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SQL_ERROR_WHILE_STORING_PROTOCOL_VERSION, e);
        }
    }

    public static boolean supportsFilteredOutLargeObjectsStatusCode(Session session) {
        return !getUsedVersion(session).isOlder(_SUPPORTS_FILTERED_OUT_LARGE_OBJECTS_STATUS_CODE.getVersion(), 3);
    }

    public static boolean supportsAttachmentStreaming(Session session) {
        return !getUsedVersion(session).isOlder(SUPPORTS_ATTACHMENT_STREAMING.getVersion(), 3);
    }

    public static boolean supportsContactImageStreaming(Session session) {
        return !getUsedVersion(session).isOlder(SUPPORTS_CONTACT_IMAGE_STREAMING.getVersion(), 3);
    }

    public static boolean supportsStandardFolderSwapping(Session session) {
        return !getUsedVersion(session).isOlder(SUPPORTS_STANDARD_FOLDER_SWAPPING.getVersion(), 3);
    }

    public static Version getUsedVersion(Session session) {
        return Version.fromString(session.getPersistentField(_CLIENT_USM_JSON_VERSION_KEY));
    }

    private final Version _version;

    private USMJSONVersion(String version) {
        _version = Version.fromString(version);
    }

    private Version getVersion() {
        return _version;
    }

    private boolean matches(Version requestedVersion) {
        return requestedVersion.getMajor() == _version.getMajor() && requestedVersion.getMinor() == _version.getMinor();
    }

    private boolean supports(Version requestedVersion) {
        return requestedVersion.getMajor() == _version.getMajor() && requestedVersion.getMinor() <= _version.getMinor();
    }

    public static void verifyProtocolVersion(Session session, USMJSONVersion requiredVersion) throws USMJSONAPIException {
        Version required = requiredVersion.getVersion();
        Version used = getUsedVersion(session);
        if (required.isNewer(used, 2))
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.FEATURE_NOT_SUPPORTED,
                ResponseStatusCode.BAD_REQUEST,
                "Feature not supported with protocol version " + used + ", requires version " + required);
    }
}
