/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.json;

import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.FolderNotFoundException;
import com.openexchange.usm.api.exceptions.InvalidUUIDException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.TemporaryDownOrBusyException;
import com.openexchange.usm.api.exceptions.USMAccessDeniedException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.connector.exceptions.DataObjectNotFoundException;
import com.openexchange.usm.connector.exceptions.MultipleOperationsOnDataObjectException;
import com.openexchange.usm.json.response.ResponseStatusCode;

public class USMJSONAPIException extends USMException {
	private static final long serialVersionUID = 1L;

	public static USMJSONAPIException createInternalError(int errorCode, USMException cause) {
		if (cause instanceof USMJSONAPIException)
			return (USMJSONAPIException) cause;
		if(cause instanceof InvalidUUIDException)
		    return createInvalidUUIDException((InvalidUUIDException) cause);
        if(cause instanceof DataObjectNotFoundException)
            return createObjectNotFoundException((DataObjectNotFoundException) cause);
		ResponseStatusCode code = ResponseStatusCode.INTERNAL_ERROR;
		if (cause instanceof USMAccessDeniedException || cause instanceof AuthenticationFailedException)
			code = ResponseStatusCode.ACCESS_DENIED;
		else if (cause instanceof TemporaryDownOrBusyException)
			code = ResponseStatusCode.TEMPORARY_NOT_AVAILABLE;
		else if (cause instanceof OXCommunicationException)
			code = ResponseStatusCode.OX_SERVER_ERROR;
		else if (cause instanceof StorageAccessException || cause instanceof USMStorageException)
			code = ResponseStatusCode.DATABASE_ERROR;
		else if (cause instanceof FolderNotFoundException)
			code = ResponseStatusCode.FOLDER_NOT_FOUND;
		return new USMJSONAPIException(errorCode, code, cause.getErrorMessage(), cause.getOxErrorForJSONResponse(),
				cause.getErrorDetailsForJSONResponse(), cause);
	}

    public static USMJSONAPIException createInvalidUUIDException(InvalidUUIDException e) {
        return new USMJSONAPIException(e.getErrorCode(), ResponseStatusCode.WRONG_MISSING_PARAMETERS, e.getMessage());
    }

    public static USMJSONAPIException createObjectNotFoundException(DataObjectNotFoundException e) {
        return new USMJSONAPIException(e.getErrorCode(), ResponseStatusCode.WRONG_MISSING_PARAMETERS, e.getMessage());
    }

    public static USMJSONAPIException createMultipleOperationsOnDataObjectException(MultipleOperationsOnDataObjectException e) {
        return new USMJSONAPIException(e.getErrorCode(), ResponseStatusCode.WRONG_MISSING_PARAMETERS, e.getMessage());
    }
	
	public static USMJSONAPIException createJSONError(int errorCode, JSONException e) {
		JSONObject errorDetails = new JSONObject();
		try {
			errorDetails.put("JSONError", e.getMessage());
		} catch (JSONException ignored) {
			// Should never be possible, ignored to avoid endless error recursion
		}
		return new USMJSONAPIException(errorCode, ResponseStatusCode.INTERNAL_ERROR, "Error creating JSON data", null,
				errorDetails, e);
	}

	private final JSONObject _oxError;

	private final JSONObject _errorDetails;

	private final ResponseStatusCode _status;

	public USMJSONAPIException(int errorCode, ResponseStatusCode status, String message) {
		this(errorCode, status, message, null, null, null);
	}

	public USMJSONAPIException(int errorCode, ResponseStatusCode status, String message, JSONObject errorDetails) {
		this(errorCode, status, message, null, errorDetails, null);
	}

	public USMJSONAPIException(int errorCode, ResponseStatusCode status, String message, Throwable cause) {
		this(errorCode, status, message, null, null, cause);
	}


	public USMJSONAPIException(int errorCode, ResponseStatusCode status, String message, JSONObject oxError,
			JSONObject errorDetails, Throwable cause) {
		super(errorCode, message, cause);
		if (errorDetails == null && (cause instanceof OXCommunicationException))
			errorDetails = ((OXCommunicationException) cause).getOxErrorForJSONResponse();
		if (errorDetails == null && status == ResponseStatusCode.INTERNAL_ERROR && cause != null) {
			errorDetails = USMException.appendStacktrace(cause, new JSONObject());
		}
		_oxError = oxError;
		_errorDetails = errorDetails;
		_status = status;
	}

	@Override
	public JSONObject getOxErrorForJSONResponse() {
		return _oxError;
	}

	@Override
	public JSONObject getErrorDetailsForJSONResponse() {
		return _errorDetails;
	}

	public ResponseStatusCode getResponseStatusCode() {
		return _status;
	}
}
