/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.json;

import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import javax.servlet.http.HttpSession;
import com.openexchange.usm.json.response.ResponseStatusCode;

/**
 * {@link HttpSessionUtil} Helper class so that USM can work in standalone mode with a servlet container that doesn't support attributes of
 * HTTP sessions.
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class HttpSessionUtil {

    private static final boolean _DISABLE_STANDALONE_SUPPORT = false;

    private static final Map<String, USMSessionCredentials> _SESSIONS = new ConcurrentHashMap<String, USMSessionCredentials>(16, 0.75f, 1);

    private HttpSessionUtil() {
    }

    public static void storeSession(HttpSession httpSession, String randomHttpSessionID, USMSessionCredentials usmSessionID) {
        try {
            httpSession.setAttribute(randomHttpSessionID, usmSessionID); // save the USM session in the HttpSession
        } catch (UnsupportedOperationException e) {
            if (_DISABLE_STANDALONE_SUPPORT)
                throw e;
            _SESSIONS.put(randomHttpSessionID, usmSessionID);
        }
    }

    public static USMSessionCredentials getSession(HttpSession httpSession, String sessionID) throws USMJSONAPIException {
        USMSessionCredentials creds = findSession(httpSession, sessionID);
        if (creds != null)
            return creds;
        throw new USMJSONAPIException(
            ConnectorBundleErrorCodes.COMMAND_UNKNOWN_SESSION,
            ResponseStatusCode.UNKNOWN_SESSION,
            "Unknown " + SESSIONID);
    }

    private static USMSessionCredentials findSession(HttpSession httpSession, String sessionID) {
        try {
            return (USMSessionCredentials) httpSession.getAttribute(sessionID);
        } catch (UnsupportedOperationException e) {
            if (_DISABLE_STANDALONE_SUPPORT)
                throw e;
            return _SESSIONS.get(sessionID);
        }
    }

    public static void removeSession(HttpSession httpSession, String sessionID) {
        try {
            httpSession.removeAttribute(sessionID);
        } catch (UnsupportedOperationException e) {
            if (_DISABLE_STANDALONE_SUPPORT)
                throw e;
            _SESSIONS.remove(sessionID);
        }
    }
}
