/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.json;

import java.io.IOException;
import java.io.OutputStream;

/**
 * {@link ByteCountOutputStream}: Simple OutputStream that does not write the data anywhere, but only counts how many bytes would be
 * written.
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class ByteCountOutputStream extends OutputStream {

    private long _byteCount = 0;

    /*
     * (non-Javadoc)
     * @see java.io.OutputStream#write(int)
     */
    @Override
    public void write(int b) throws IOException {
        _byteCount++;
    }

    @Override
    public void write(byte[] b) throws IOException {
        _byteCount += b.length;
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        _byteCount += len;
    }

    /**
     * @return Number of bytes that so far have been written to this ByteCountOutputStream.
     */
    public long getByteCount() {
        return _byteCount;
    }
}
