/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.connector.sync;

import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.DataObjectFilter;
import com.openexchange.usm.api.session.assets.ChangeState;

public class PIMAtachmentsCountFilter implements DataObjectFilter {

    private final long _originalSyncID;

    private final int _maxCount;

    private boolean _blocked;

    private int _attachmentCount;

    public PIMAtachmentsCountFilter(long originalSyncID, int maxCount) {
        _originalSyncID = originalSyncID;
        _maxCount = maxCount;
    }

    @Override
    public void initialize() {
        _attachmentCount = 0;
    }

    @Override
    public boolean accept(DataObject object) {
        return true;
    }

    @Override
    public boolean addOperationToResult(ChangeState action, DataObject object) {
        if (_blocked || object == null)
            return false;
        switch (action) {
        case CREATED:
        case MODIFIED:
            int count = getAttachmentCount(object);
            if (_attachmentCount > 0 && _attachmentCount + count > _maxCount) {
                _blocked = true;
                return false;
            }
            _attachmentCount += count;
            return true;
        default:
            return true;
        }
    }

    private int getAttachmentCount(DataObject object) {
        try {
            Object o = object.getFieldContent("number_of_attachments");
            return (o instanceof Number) ? ((Number) o).intValue() : 0;
        } catch (IllegalArgumentException ignored) {
            return 0;
        }
    }

    @Override
    public long getOriginalSyncID() {
        return _originalSyncID;
    }
}
