/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.HOST;
import static com.openexchange.usm.connector.commands.CommandConstants.PORT;
import static com.openexchange.usm.connector.commands.CommandConstants.PUSH_TYPE_NONE;
import static com.openexchange.usm.connector.commands.CommandConstants.PUSH_TYPE_UDP;
import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import static com.openexchange.usm.connector.commands.CommandConstants.TYPE;
import java.net.InetAddress;
import java.net.UnknownHostException;
import javax.servlet.http.HttpServletRequest;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.USMSessionCredentials;
import com.openexchange.usm.json.push.PushNotificationService;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.util.Toolkit;

/**
 * Handler for the SetPushNotification USM-JSON-Command.
 * @author ldo
 *
 */
public class SetPushNotificationHandler extends NormalCommandHandler {
	private static final String[] REQUIRED_PARAMETERS = { SESSIONID, TYPE };
	private static final String[] OPTIONAL_PARAMETERS = { HOST, PORT };

    public SetPushNotificationHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
    }

	@Override
	public ResponseObject handleRequest() throws USMJSONAPIException {
		USMSessionCredentials usmSessionID = (USMSessionCredentials) getHttpSession().getAttribute(getStringParameter(SESSIONID));
		String type = getStringParameter(TYPE);
		//DEBUG 
		// the service must be started by initialization
		PushNotificationService.getInstance().startup();
		if (PUSH_TYPE_NONE.equals(type)) {
			//  find and remove tapeInfo for key usmSessionID from  the map
			if (PushNotificationService.getInstance().isRun()) {
				if (PushNotificationService.getInstance().containKey(usmSessionID)) {
					PushNotificationType ti = PushNotificationService.getInstance().removeValue(usmSessionID);
					_session.removeChangeListener(ti);
				}
				return new ResponseObject(ResponseStatusCode.SUCCESS);
			}
			return new ResponseObject(ResponseStatusCode.INTERNAL_ERROR, null,
					ConnectorBundleErrorCodes.SET_PUSH_NOTIFICATION_SERVER_ISNOT_RUN_ERROR,
					"the push notification service is't running", null, null, null);
		} else if (PUSH_TYPE_UDP.equals(type)) {
			String host = getStringParameter(HOST);
			if (isHostInvalid(host))
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.SET_PUSH_NOTIFICATION_INVALID_HOST_ERROR,
						ResponseStatusCode.WRONG_MISSING_PARAMETERS, "invalid host parameter for SetPushNotification");

			int portInt = convertToInt(getStringParameter(PORT));
			if (portInt < 0)
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.SET_PUSH_NOTIFICATION_INVALID_PORT_ERROR,
						ResponseStatusCode.WRONG_MISSING_PARAMETERS, "invalid port parameter for SetPushNotification");
			PushNotificationTypeUDP typeInfo = new PushNotificationTypeUDP(host, portInt, _session);

			// set typeInfo by service with key usmSessionID
			if (PushNotificationService.getInstance().isRun()) {
				_session.addChangeListener(typeInfo);
				PushNotificationService.getInstance().setValue(usmSessionID, typeInfo);
				return new ResponseObject(ResponseStatusCode.SUCCESS);
			}
			return new ResponseObject(ResponseStatusCode.INTERNAL_ERROR, null,
					ConnectorBundleErrorCodes.SET_PUSH_NOTIFICATION_SERVER_ISNOT_RUN_ERROR,
					"the push notification service is't running", null, null, null);
		} else
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.SET_PUSH_NOTIFICATION_INVALID_TYPE_ERROR,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "invalid type parameter for SetPushNotification");
	}

	@Override
	protected String[] getOptionalParameters() {
		return OPTIONAL_PARAMETERS;
	}

	@Override
	protected String[] getRequiredParameters() {
		return REQUIRED_PARAMETERS;
	}

	private int convertToInt(String port) {
		if (Toolkit.isNullOrEmpty(port))
			return -1;
		try {
			return Integer.valueOf(port);
		} catch (NumberFormatException e) {
			return -1;
		}
	}

	private boolean isHostInvalid(String host) {

		try {
			InetAddress.getByName(host);
		} catch (UnknownHostException e) {
			return true;
		}

		return false;
	}
}
//TODO: use com.openexchange.push.udp.RegisterHandler.addRegisterObject(registerObject) where registerObject - com.openexchange.push.udp.RegisterObject