/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import static com.openexchange.usm.connector.commands.CommandConstants.UID;
import static com.openexchange.usm.datatypes.tasks.calendar.UserParticipantObject.CONFIRMATION;
import static com.openexchange.usm.datatypes.tasks.calendar.UserParticipantObject.CONFIRM_MESSAGE;
import javax.servlet.http.HttpServletRequest;
import com.openexchange.usm.api.contenttypes.calendar.AppointmentContentType;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.folder.FolderConstants;
import com.openexchange.usm.api.contenttypes.task.TaskContentType;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.assets.SyncResult;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;

/**
 * Handler for the USM-setConfirmationStatus command.
 * 
 * @author tse
 */

public class SetConfirmationStatusHandler extends NormalCommandHandler {

    private static final String[] REQUIRED_PARAMETERS = { SESSIONID, UID, CONFIRMATION, CONFIRM_MESSAGE };

    private static final String[] OPTIONAL_PARAMETERS = {};

    // see http://oxpedia.org/index.php?title=HTTP_API#DefaultTypes
    private static final Integer STANDARD_FOLDER_TYPE_CALENDAR = 2;

    private static final Integer STANDARD_FOLDER_TYPE_TASK = 1;

    public SetConfirmationStatusHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
    }

    @Override
    public ResponseObject handleRequest() throws USMJSONAPIException {
        try {
            return internalHandleRequest();
        } catch (StorageAccessException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SET_CONFIRMATION_STATUS_DB_ERROR, e);
        } catch (USMStorageException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SET_CONFIRMATION_STATUS_SQL_ERROR, e);
        } catch (AuthenticationFailedException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SET_CONFIRMATION_STATUS_AUTHENTICATION_ERROR, e);
        } catch (OXCommunicationException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SET_CONFIRMATION_STATUS_OX_COMMUNICATION_ERROR, e);
        }
    }

    private ResponseObject internalHandleRequest() throws USMJSONAPIException, StorageAccessException, USMStorageException, AuthenticationFailedException, OXCommunicationException {

        String appointmentOrTaskUid = getString(_parameters, UID);
        int confirmation = getInt(_parameters, CONFIRMATION);
        String confirmMsg = getString(_parameters, CONFIRM_MESSAGE);

        DataObject appointmentOrTask = findObject(appointmentOrTaskUid);
        setConfirmationStatus(appointmentOrTask, confirmation, confirmMsg);
        return new ResponseObject(ResponseStatusCode.SUCCESS);

    }

    private Folder findStandardFolder(Integer type) throws StorageAccessException, USMStorageException, USMJSONAPIException {
        Folder[] allFolders = _session.getCachedFolders();
        if (allFolders == null)
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.SET_CONFIRMATION_STATUS_NO_FOLDERS,
                ResponseStatusCode.BAD_REQUEST,
                "No folder hierarchy available. Please synchronize the folder hierarchy.");

        for (Folder folder : allFolders) {
            Object isStandardFolder = folder.getFieldContent(FolderConstants.STANDARD_FOLDER);
            if (Boolean.TRUE.equals(isStandardFolder)) {
                Object standardFolderType = folder.getFieldContent(FolderConstants.STANDARD_FOLDER_TYPE);
                if (type.equals(standardFolderType))
                    return folder;
            }
        }
        throw new USMJSONAPIException(
            ConnectorBundleErrorCodes.SET_CONFIRMATION_STATUS_NO_STANDARD_CALENDAR_FOLDER,
            ResponseStatusCode.INTERNAL_ERROR,
            "No standard calendar/tasks folder found.");
    }

    private Folder findStandardCalendarFolder() throws StorageAccessException, USMStorageException, USMJSONAPIException {
        return findStandardFolder(STANDARD_FOLDER_TYPE_CALENDAR);
    }

    private Folder findStandardTasksFolder() throws StorageAccessException, USMStorageException, USMJSONAPIException {
        return findStandardFolder(STANDARD_FOLDER_TYPE_TASK);
    }

    private DataObject findObject(String objectUID) throws StorageAccessException, USMStorageException, USMJSONAPIException {
        Folder standardFolder = findStandardCalendarFolder();
        DataObject appointment = findObjectInStandardFolder(standardFolder, objectUID);
        if (appointment != null)
            return appointment;

        standardFolder = findStandardTasksFolder();
        DataObject task = findObjectInStandardFolder(standardFolder, objectUID);
        if (task != null)
            return task;

        // no appointment found
        throw new USMJSONAPIException(
            ConnectorBundleErrorCodes.SET_CONFIRMATION_STATUS_NO_OBJECT_FOUND_FOR_UID,
            ResponseStatusCode.UNKNOWN_UID,
            "No appointment/task found for UID '" + objectUID + "'");
    }

    private DataObject findObjectInStandardFolder(Folder folder, String objectUID) throws StorageAccessException, USMStorageException, USMJSONAPIException {

        // search in cached data
        DataObject[] elements = _session.getCachedFolderElements(folder.getID(), folder.getElementsContentType());
        DataObject appointment = findByUid(elements, objectUID);
        if (appointment != null)
            return appointment;

        // perform a sync (without storing the sync result) to see if the appointment is there
        try {
            SyncResult syncResult = _session.syncWithServer(folder.getID(), Session.NO_LIMIT, null, false, null);
            appointment = findByUid(syncResult.getNewState(), objectUID);
            if (appointment != null)
                return appointment;
        } catch (USMException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SET_CONFIRMATION_STATUS_SYNC_EXCEPTION, e);
        }
        return null;

    }

    private DataObject findByUid(DataObject[] elements, String uid) {
        if (elements == null)
            return null;
        for (DataObject element : elements) {
            Object elementUid = element.getFieldContent(UID);
            if (uid.equals(elementUid)) {
                return element;
            }
        }
        return null;
    }

    private void setConfirmationStatus(DataObject appointmentOrTask, int confirmation, String confirmMsg) throws AuthenticationFailedException, OXCommunicationException, USMJSONAPIException {
        ContentType contentType = appointmentOrTask.getContentType();
        if (contentType instanceof AppointmentContentType) {
            ((AppointmentContentType) contentType).confirm(appointmentOrTask, confirmation, confirmMsg);
        } else if (contentType instanceof TaskContentType) {
            ((TaskContentType) contentType).confirm(appointmentOrTask, confirmation, confirmMsg);
        } else {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.SET_CONFIRMATION_STATUS_INVALID_CONTENT_TYPE,
                ResponseStatusCode.BAD_REQUEST,
                "Invalid content type " + contentType.getID());
        }
    }

    @Override
    protected String[] getOptionalParameters() {
        return OPTIONAL_PARAMETERS;
    }

    @Override
    protected String[] getRequiredParameters() {
        return REQUIRED_PARAMETERS;
    }

}
