/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.MAIL;
import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import java.io.IOException;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.mail.MailContentType;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.io.InputStreamProvider;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.USMJSONVersion;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.mimemail.MimeMailBuilder;
import com.openexchange.usm.mimemail.StreamingMimeMailBuilder;
import com.openexchange.usm.util.TempFileStorage;

/**
 * Handler for the SendMail USM-JSON-Command.
 * 
 * @author ldo
 */
public class SendMailHandler extends NormalCommandHandler {

    private static final String[] REQUIRED_PARAMETERS = { SESSIONID, MAIL };

    private static final String[] OPTIONAL_PARAMETERS = {};

    public SendMailHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
    }

    @Override
    public ResponseObject handleRequest() throws USMJSONAPIException {

        JSONObject data = getJSONObject(_parameters, MAIL);
        if (data == null)
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.SEND_MAIL_MISSING_DATA_ERROR,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "No mail provided to SendMail");
        if (USMJSONVersion.supportsAttachmentStreaming(_session)) {
            StreamingMimeMailBuilder mimeMailBuilder = new StreamingMimeMailBuilder(_session);
            InputStreamProvider streamProvider = mimeMailBuilder.convertJSONObjectToMimeMail(data);
            MailContentType mailContentType = (MailContentType) _servlet.getContentTypeManager().getContentType(DefaultContentTypes.MAIL_ID);
            try {
                // send mail to server
                mailContentType.sendMail(streamProvider, _session);
            } catch (USMException e) {
                throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SEND_MAIL_SENDMAIL_ACTION_ERROR, e);
            } finally {
                try {
                    TempFileStorage.deleteTempFiles(_session, mimeMailBuilder.getTempIDs());
                } catch (IOException e) {
                    _servlet.getJournal().debug("Error: deleting temporary files failed! " + e);
                }
            }
        } else {
            MimeMailBuilder mimeMailBuilder = new MimeMailBuilder();
            // convert JSON structure to mime mail
            String mimeMail = mimeMailBuilder.convertJSONObjectToMimeMail(data);
            MailContentType mailContentType = (MailContentType) _servlet.getContentTypeManager().getContentType(DefaultContentTypes.MAIL_ID);
            try {
                // send mail to server
                mailContentType.sendMail(mimeMail, _session);
            } catch (USMException e) {
                throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SEND_MAIL_SENDMAIL_ACTION_ERROR, e);
            }
        }
        return new ResponseObject(ResponseStatusCode.SUCCESS);
    }

    @Override
    protected String[] getOptionalParameters() {
        return OPTIONAL_PARAMETERS;
    }

    @Override
    protected String[] getRequiredParameters() {
        return REQUIRED_PARAMETERS;
    }

    @Override
    public void disposeResources() {
        // bug fix 36718: commands which may run in parallel shall not delete all files!
        // if (_session != null)
        //    TempFileStorage.deleteTempFiles(_session);
    }
}
