/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import static com.openexchange.usm.connector.commands.CommandConstants.TEMPID;
import java.io.IOException;
import javax.servlet.http.HttpServletRequest;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.USMJSONVersion;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.util.TempFileStorage;

/**
 * {@link ClearDataHandler} Allows the client to explicitly remove 1 or all previously created (with uploadData) temporary data files
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class ClearDataHandler extends NormalCommandHandler {

    private static final String[] REQUIRED_PARAMETERS = { SESSIONID };

    private static final String[] OPTIONAL_PARAMETERS = { TEMPID };

    public ClearDataHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
        USMJSONVersion.verifyProtocolVersion(_session, USMJSONVersion.SUPPORTS_ATTACHMENT_STREAMING);
    }

    @Override
    public ResponseObject handleRequest() throws USMJSONAPIException {
        String tempid = getStringParameter(TEMPID, null);
        try {
            if (tempid != null)
                TempFileStorage.deleteTempFile(_session, tempid);
            // no else case
            // bug fix 36718: commands which may run in parallel shall not delete all files!
                
            return new ResponseObject(ResponseStatusCode.SUCCESS);
        } catch (IOException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_CLEAR_DATA_IO_ERROR_ON_DELETING_TEMP_FILE,
                ResponseStatusCode.OX_SERVER_ERROR,
                "I/O error deleting tempfile '" + tempid + '\'',
                e);
        }
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.connector.commands.BaseCommandHandler#getRequiredParameters()
     */
    @Override
    protected String[] getRequiredParameters() {
        return REQUIRED_PARAMETERS;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.connector.commands.BaseCommandHandler#getOptionalParameters()
     */
    @Override
    protected String[] getOptionalParameters() {
        return OPTIONAL_PARAMETERS;
    }
}
