/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.database.ox.update;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.update.PerformParameters;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.database.ox.USMDatabaseOXErrorCodes;
import com.openexchange.usm.database.ox.USMUpdateExceptionCodes;
import com.openexchange.usm.database.ox.USMUpdateTaskProvider;
import com.openexchange.usm.util.SQLToolkit;
import com.openexchange.usm.util.Toolkit;

public class USMDeleteStoredProceduresUpdateTaskV2 extends USMBaseUpdateTaskV2 {

    private static final Log LOG = LogFactory.getLog(USMDeleteStoredProceduresUpdateTaskV2.class);

    public final static String[] SEQUENCE_TABLES_NAMES = { "sequence_usmSessionId", "sequence_usmIdMapping" };

	public USMDeleteStoredProceduresUpdateTaskV2(USMUpdateTaskProvider provider) {
		super(provider);
	}

	@Override
	protected void doPerform(PerformParameters params) throws OXException {
        Connection con = params.getConnection();
        for (int i = 0; i < SEQUENCE_TABLES_NAMES.length; i++) {
            try {
                String tablename = SEQUENCE_TABLES_NAMES[i];
                if (procedureExists(tablename, con.getMetaData())) {
                    dropSequence(tablename, con, true);
                }
            } catch (USMStorageException e) {
                throw USMUpdateExceptionCodes.SQL_ERROR.create(e);
            } catch (SQLException e) {
                throw USMUpdateExceptionCodes.SQL_ERROR.create(e);
            }
        }
    }

    private static void dropSequence(String sequenceName, Connection con, boolean isAutocommit) throws USMStorageException {
        Statement stmt = null;
        try {
            try {
                stmt = con.createStatement();
                if (isAutocommit) {
                    con.setAutoCommit(false);
                }
                stmt.execute(SQLToolkit.dropSequenceProcedure(sequenceName));
                con.commit();
                LOG.info("Dropping of sequence " + sequenceName + " successfully performed");
            } catch (SQLException e) {
                if (isAutocommit) {
                    con.rollback();
                }
                if (e.getMessage().contains("alter routine command denied to user")) {
                    LOG.info("Dropping of sequence " + sequenceName + " was not possible. Probably it did not exist anyway.");
                } else {
                    throw new USMStorageException(USMDatabaseOXErrorCodes.CREATE_SEQUENCE_ERROR, e.getMessage(), e);
                }
            } finally {
                if (isAutocommit) {
                    con.setAutoCommit(true);
                }
                Toolkit.close(stmt);
            }
        } catch (SQLException e) {
            throw new USMStorageException(USMDatabaseOXErrorCodes.CREATE_SEQUENCE_ERROR_AUTOCOMMIT_SETTINGS, e.getMessage(), e);
        }
    }

    private static boolean procedureExists(final String procedureName, final DatabaseMetaData dbmd) throws SQLException {
        ResultSet resultSet = null;
        try {
            resultSet = dbmd.getProcedures(null, null, "get_" + procedureName + "_id");
            if (resultSet == null) {
                return false;
            }
            return resultSet.next();
        } finally {
            Toolkit.close(resultSet);
        }
    }

    @Override
    public String[] getDependencies() {
		return new String[] { USMTablesUpdateTaskV2.class.getName() };
	}
}
