/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.database.ox.update;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.update.PerformParameters;
import com.openexchange.tools.update.Tools;
import com.openexchange.usm.database.ox.USMUpdateExceptionCodes;
import com.openexchange.usm.database.ox.USMUpdateTaskProvider;

/**
 * {@link USMCreatePrimaryKeyUpdateTask} Replaces unique index with primary key for those USM table for which the primary key was missing.
 * Adds primary key or removes unique index if a previous execution was partially successful.
 *
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class USMCreatePrimaryKeyUpdateTask extends USMBaseUpdateTaskV2 {

    private static final Log LOG = LogFactory.getLog(USMCreatePrimaryKeyUpdateTask.class);

    private static class PrimaryKeyMappingInfo {

        public final String _tableName;

        public final String _indexName;

        public final String _primaryKeyColumns;

        public PrimaryKeyMappingInfo(String table, String index, String primaryKeyColumns) {
            _tableName = table;
            _indexName = index;
            _primaryKeyColumns = primaryKeyColumns;
        }
    }

    private static final PrimaryKeyMappingInfo[] _TABLES_WITHOUT_PRIMARY_KEY = {
        new PrimaryKeyMappingInfo("usmIdMapping", "usmIdMappingUIdx", "cid, usmSessionId, shortId"),
        new PrimaryKeyMappingInfo("easUIDMapping", "easUIDMappingUIDX", "cid, usmSessionId, folderId, shortId"),
        new PrimaryKeyMappingInfo("USMDataStorage", "USMDataStorageUIDX", "cid, usmSessionId, objectId, syncKey"),
        new PrimaryKeyMappingInfo("USMSessionFields", "cid", "cid, usmSessionId, field") };

    public USMCreatePrimaryKeyUpdateTask(USMUpdateTaskProvider provider) {
        super(provider);
    }

    @Override
    public String[] getDependencies() {
        return new String[] { USMTablesUpdateTaskV2.class.getName(), USMUUIDTablesUpdateTaskV2.class.getName() };
    }

    @Override
    protected void doPerform(PerformParameters params) throws OXException {
        Connection con = params.getConnection();
        Statement stmt = null;
        try {
            con.setAutoCommit(false);
            for (PrimaryKeyMappingInfo info : _TABLES_WITHOUT_PRIMARY_KEY) {
                stmt = dropIndexAndAddPrimaryKey(con, stmt, info);
            }
            con.commit();
        } catch (SQLException e) {
            rollback(con);
            throw USMUpdateExceptionCodes.SQL_ERROR.create(e);
        } finally {
            enableAutocommit(con, stmt);
        }
    }

    private static Statement dropIndexAndAddPrimaryKey(Connection con, Statement stmt, PrimaryKeyMappingInfo info) throws SQLException {
        Set<String> indexes = USMBaseUpdateTaskV2.getTableIndexes(info._tableName, con);
        String alterTableOperations = buildAlterTableOperations(info, Tools.hasPrimaryKey(con, info._tableName), indexes);
        if (alterTableOperations != null) {
            if (stmt == null) {
                stmt = con.createStatement();
            }
            String alterTableStatement = "ALTER TABLE " + info._tableName + ' ' + alterTableOperations;
            LOG.debug(alterTableStatement);
            stmt.execute(alterTableStatement);
        }
        return stmt;
    }

    private static String buildAlterTableOperations(PrimaryKeyMappingInfo info, boolean hasPrimaryKey, Set<String> indexes) {
        if (indexes.contains(info._indexName)) {
            if (hasPrimaryKey) {
                return "DROP INDEX " + info._indexName;
            }
            return "ADD PRIMARY KEY(" + info._primaryKeyColumns + "), DROP INDEX " + info._indexName;
        }
        if (!hasPrimaryKey) {
            return "ADD PRIMARY KEY(" + info._primaryKeyColumns + ')';
        }
        return null;
    }
}
