/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.database.ox.update;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.update.NamespaceAwareUpdateTask;
import com.openexchange.groupware.update.PerformParameters;
import com.openexchange.usm.database.ox.TableDefinition;
import com.openexchange.usm.database.ox.USMUpdateExceptionCodes;
import com.openexchange.usm.database.ox.USMUpdateTaskProvider;

@NamespaceAwareUpdateTask(namespace = "groupware.utf8mb4", description = "All utf8 to utf8mb4 conversion tasks")
public class USMChangeCollationToUtf8mb4UpdateTaskV2 extends USMBaseUpdateTaskV2 {

	public USMChangeCollationToUtf8mb4UpdateTaskV2(USMUpdateTaskProvider provider) {
		super(provider);
	}

	@Override
    public String[] getDependencies() {
		return new String[] { USMTablesUpdateTaskV2.class.getName() };
	}

	@Override
	protected void doPerform(PerformParameters params) throws OXException {
		Connection con = params.getConnection();
		Statement stmt = null;
		try {
			con.setAutoCommit(false);
			stmt = con.createStatement();
			for (String table : TableDefinition.TABLE_NAMES) {
                alterCollationForTable(table, stmt);
            }
			for (String table : USMUUIDTablesUpdateTaskV2.TABLE_NAMES) {
                alterCollationForTable(table, stmt);
            }
			con.commit();
		} catch (SQLException e) {
			rollback(con);
			throw USMUpdateExceptionCodes.SQL_ERROR.create(e);
		} finally {
			enableAutocommit(con, stmt);
		}
	}

	private void alterCollationForTable(String tableName, Statement stmt) throws SQLException {
		stmt.executeUpdate("ALTER TABLE " + tableName + " CONVERT TO CHARACTER SET utf8mb4 COLLATE utf8mb4_bin");
	}
}
