/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.contenttypes.util;

import java.io.Serializable;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.datatypes.DataType;
import com.openexchange.usm.api.exceptions.DeserializationFailedException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.session.dataobject.SimpleDataObject;

/**
 * Abstract base class for typical ContentType implementations.
 * 
 * @author afe
 */
public abstract class AbstractContentType implements ContentType {

    private static final int IDENTICAL_RATING_VALUE = 5;

    private static final int SIMILAR_RATING_VALUE = 2;

    private static final int FIELD_MATCH_PERCENTAGE = 50; // from 0 to 100

    /*
     * This method can be overwritten by ContentTypes to provide ContentType-specific match algorithms (non-Javadoc)
     * @see com.openexchange.usm.api.contenttypes.ContentType#getMatchRating(com.openexchange.usm.api.session.DataObject,
     * com.openexchange.usm.api.session.DataObject)
     */
    @Override
    public int getMatchRating(DataObject client, DataObject server) {
        // TODO Add some default heuristic to determine possible match ?
        return -1;
    }

    /**
     * Default method of matching content types, can be called by sub-classes in the getMatchRating() method if no special handling is
     * implemented.
     * 
     * @param client
     * @param server
     * @return
     */
    protected int getDefaultMatchRating(DataObject client, DataObject server) {
        int matchingFields = 0;
        int nonEmptyFields = 0;
        boolean forceMatchRating = false;
        int matchValue = 0;
        ContentTypeField[] fields = getFields();
        for (int i = 0; i < fields.length; i++) {
            ContentTypeField field = fields[i];
            int weight = field.getFieldWeight();
            if (weight != 0) {
                DataType<?> dataType = field.getFieldType();
                Object value1 = client.getFieldContent(i);
                Object value2 = server.getFieldContent(i);
                if (!dataType.isDefaultValue(value1) || !dataType.isDefaultValue(value2)) {
                    nonEmptyFields++;
                    int factor = performSingleMatch(dataType, value1, value2);
                    if (factor > 0) {
                        if (weight < 0) {
                            forceMatchRating = true;
                            weight = -weight;
                        }
                        matchingFields++;
                        matchValue += factor * weight;
                    }
                }
            }
        }
        return (forceMatchRating || nonEmptyFields * FIELD_MATCH_PERCENTAGE <= matchingFields * 100) ? matchValue : -1;
    }

    /**
     * Performs a single match between two field values. At least one of the two values is not at the default value for the DataType
     * 
     * @param dataType
     * @param value1
     * @param value2
     * @return 0 if no match, value > 0 indicating the quality of the match
     */
    protected int performSingleMatch(DataType<?> dataType, Object value1, Object value2) {
        if (dataType.isEqual(value1, value2))
            return IDENTICAL_RATING_VALUE;
        // TODO Do we want to delegate this to the DataType ?
        if (value1 != null && value2 != null && value1.toString().trim().equalsIgnoreCase(value2.toString().trim()))
            return SIMILAR_RATING_VALUE;
        return 0;
    }

    @Override
    public DataObject newDataObject(Session session) {
        return new SimpleDataObject(session, this);
    }

    /**
     * Default implementation of deserializeData() that automatically handles the addition of new ContentTypeFields at the end of the
     * ContentType field list by setting them to their default value if not present in the deserialized data.
     */
    @Override
    public void deserializeData(Serializable[] data, int offset, Object[] fieldContent) throws DeserializationFailedException {
        ContentTypeField[] fields = getFields();
        int limit = Math.min(data.length - offset, fieldContent.length);
        int i = 0;
        try {
            for (; i < limit; i++)
                fieldContent[i] = fields[i].getFieldType().deserialize(data[i + offset]);
        } catch (Exception e) {
            throw new DeserializationFailedException(
                USMContentTypesUtilErrorCodes.DESERIALIZATION_FAILED_INVALID_FIELD_CONTENT,
                "Invalid serialized data, probably for field " + fields[i].getFieldName(),
                e);
        }
        for (; i < fieldContent.length; i++)
            fieldContent[i] = fields[i].getFieldType().createNewObject();
    }

    @Override
    public boolean canBeFolderElementsContentType() {
        return true;
    }
}
