/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.datatypes.tasks.calendar;

import java.io.Serializable;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Defines Confirming participant for the Calendar and Tasks module.
 * 
 * @author ldo
 */
public class ConfirmingParticipant implements Serializable {

    private static final long serialVersionUID = 1L;

    static final String TYPE = "type";

    static final String MAIL = "mail";

    static final String DISPLAY_NAME = "display_name";

    static final String STATUS = "status";

    static final String CONFIRM_MESSAGE = "confirmmessage";

    /** Type of the participant. 1- user; 5- external user */
    private final int _type;

    /** email address of external participant */
    private final String _mail;

    /** display name of external participant */
    private final String _displayName;

    /** Valid values: 0-3 */
    private final int _status;

    /** Confirm Message of the participant */
    private final String _confirmMessage;

    public ConfirmingParticipant(int type, String mail, String displayName, int status, String confirmMessage) {
        _type = type;
        _mail = mail;
        _displayName = displayName;
        _status = status;
        _confirmMessage = confirmMessage;
    }

    public ConfirmingParticipant() {
        _type = 0;
        _mail = "";
        _displayName = "";
        _status = 0;
        _confirmMessage = "";
    }

    public ConfirmingParticipant(JSONObject jObj) {
        _type = jObj.optInt(TYPE, 0);
        _mail = jObj.optString(MAIL, "");
        _displayName = jObj.optString(DISPLAY_NAME, "");
        _status = jObj.optInt(STATUS, 0);
        _confirmMessage = jObj.optString(CONFIRM_MESSAGE, "");
    }

    public int getType() {
        return _type;
    }

    public String getMail() {
        return _mail;
    }

    public String getDisplayName() {
        return _displayName;
    }

    public int getStatus() {
        return _status;
    }

    public String getConfirmMessage() {
        return _confirmMessage;
    }

    public JSONObject toJSONObject() throws JSONException {
        JSONObject result = new JSONObject();
        result.put(TYPE, _type);
        result.put(MAIL, _mail);
        result.put(DISPLAY_NAME, _displayName);
        result.put(STATUS, _status);
        if (!_confirmMessage.isEmpty()) {
            result.put(CONFIRM_MESSAGE, _confirmMessage);
        }
        return result;
    }

    @Override
    public String toString() {
        return '{' + TYPE + ':' + String.valueOf(_type) + ';' + MAIL + ':' + _mail + ';' + DISPLAY_NAME + ':' + _displayName + ';' + STATUS + ':' + String.valueOf(_status) + ';' + CONFIRM_MESSAGE + ':' + _confirmMessage + '}';
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + _type;
        result = prime * result + _mail.hashCode();
        result = prime * result + _displayName.hashCode();
        result = prime * result + _status;
        result = prime * result + _confirmMessage.hashCode();
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        ConfirmingParticipant other = (ConfirmingParticipant) obj;
        if (_type != other.getType())
            return false;
        if (!_mail.equals(other.getMail()))
            return false;
        if (!_displayName.equals(other.getDisplayName()))
            return false;
        if (_status != other.getStatus())
            return false;
        if (!_confirmMessage.equals(other.getConfirmMessage()))
            return false;
        return true;
    }
}
