/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.datatypes.mail;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.contenttypes.mail.impl.MailContentTypeImpl;

public class NestedMail implements Serializable {

    private static final String FILENAME = "filename";

    private static final String ID = "id";

    private static final String FOLDER_ID = "folder_id";

    private static final String ATTACHMENT = "attachment";

    /**
	 * 
	 */
    private static final long serialVersionUID = 1L;

    private final Map<String, Object> _map;

    public NestedMail(Session session, JSONObject jSONObject) throws JSONException {
        Map<String, Object> map = new HashMap<String, Object>();
        ContentTypeField[] fields = MailContentTypeImpl.getAllFields();
        for (int i = 0; i < fields.length; i++) {
            ContentTypeField field = fields[i];
            String fieldName = field.getFieldName();
            if (!isSpecialField(fieldName) && jSONObject.has(fieldName))
                map.put(field.getFieldName(), field.getFieldType().extractFromJSONObject(session, jSONObject, fieldName));
        }
        if (jSONObject.has(FILENAME))
            map.put(FILENAME, jSONObject.getString(FILENAME));
        _map = Collections.unmodifiableMap(map);
    }

    public NestedMail(Map<String, Object> map, boolean flag) {
        Map<String, Object> m = new HashMap<String, Object>();
        ContentTypeField[] fields = MailContentTypeImpl.getAllFields();
        for (int i = 0; i < fields.length; i++) {
            String fieldName = fields[i].getFieldName();
            if (!isSpecialField(fieldName) && map.containsKey(fieldName))
                m.put(fieldName, map.get(fieldName));
        }
        if (map.containsKey(FILENAME))
            m.put(FILENAME, map.get(FILENAME));
        _map = Collections.unmodifiableMap(m);
    }

    public NestedMail(List<Object> list) {
        Map<String, Object> map = new HashMap<String, Object>();
        ContentTypeField[] fields = MailContentTypeImpl.getAllFields();
        for (int i = 0; i < fields.length; i++) {
            ContentTypeField field = fields[i];
            String fieldName = field.getFieldName();
            if (i < list.size() && !isSpecialField(fieldName)) {
                Object v = list.get(i);
                if (!field.getFieldType().isDefaultValue(v))
                    map.put(fieldName, list.get(i));
            }
        }
        _map = Collections.unmodifiableMap(map);
    }

    public NestedMail() {
        _map = Collections.emptyMap();
    }

    private boolean isSpecialField(String fieldName) {
        /* removed test on ID, because it is needed due to bug 56285 */ 
        return FOLDER_ID.equals(fieldName) || ATTACHMENT.equals(fieldName);
    }

    public Map<String, Object> getMap() {
        return _map;
    }

    @Override
    public int hashCode() {
        return _map.hashCode();
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        NestedMail other = (NestedMail) obj;

        return _map.equals(other.getMap());
    }

    public JSONObject toJSONObject(Session session) throws JSONException {
        JSONObject o = new JSONObject();
        Object v = _map.get(FILENAME);
        if (v != null)
            o.put(FILENAME, v);
        ContentTypeField[] fields = MailContentTypeImpl.getAllFields();
        for (int i = 0; i < fields.length; i++) {
            ContentTypeField field = fields[i];
            String fieldName = field.getFieldName();
            v = _map.get(fieldName);
            if (v != null)
                field.getFieldType().storeInJSONObject(session, o, fieldName, v);
        }
        return o;
    }
    
    public String getFilename() {
        Object v = _map.get(FILENAME);
        if (v != null) {
            return (String) v;
        }
        return "";
    }

    public String getId() {
        Object v = _map.get(ID);
        if (v != null) {
            return (String) v;
        }
        return "";
    }
}
