/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.datatypes.mail;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.mail.MailContentType;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.datatypes.DataType;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.assets.ChangeState;
import com.openexchange.usm.session.dataobject.DataObjectUtil;
import com.openexchange.usm.session.dataobject.SimpleDataObject;

public class MeetingRequestResponseUtil {

    public static MailAttachment getICalFromExternalAppInvitationMail(DataObject mailObject) {
        if (mailObject.getContentType().getCode() != DefaultContentTypes.MAIL_CODE)
            return null;
        // go trough all email attachments and find an iCal attachment
        Object objAttach = mailObject.getFieldContent("attachments");
        if (objAttach instanceof Object[]) {
            Object[] ar = (Object[]) objAttach;
            for (Object o : ar) {
                if (o instanceof MailAttachment) {
                    MailAttachment a = (MailAttachment) o;
                    String content_type = a.getContent_type();
                    if (content_type != null &&
                        (    // RFC 6047
                            (content_type.startsWith("text/calendar") &&
                                (content_type.toLowerCase().contains("method=request")
                                 || content_type.toLowerCase().contains("method=cancel")
                                 || content_type.toLowerCase().contains("method=reply")
                                )
                            )
                            || content_type.startsWith("application/ics")
                        )) {
                        return a;
                    }
                }
            }
        }
        return null;
    }

    public static DataObject getAppointmentFromExternalInvitation(DataObject changeObject, MailAttachment icalFromExternalAppInvitationMail, Session session) throws USMException, JSONException {

        if (changeObject.getContentType().getCode() != DefaultContentTypes.MAIL_CODE)
            return null;
        if (icalFromExternalAppInvitationMail == null)
            return null;
        String attSeqId = icalFromExternalAppInvitationMail.getId();

        return getAppointmentObjectFromInvitation(changeObject, session, attSeqId);
    }

    public static DataObject getAppointmentObjectFromInvitation(DataObject changeObject, Session session, String attSeqId) throws USMException, JSONException, StorageAccessException, USMStorageException {
        JSONObject iTipAnswer = ((MailContentType) changeObject.getContentType()).analyzeICalAttachment(changeObject, attSeqId);
        if (iTipAnswer.has("actions")) {
            JSONArray actions = iTipAnswer.getJSONArray("actions");
            if (actions.length()>0) {
                // expecting actions=["ignore"]
                // no documentation available
                if ("ignore".equals(actions.get(0).toString()))
                    return null;
            }
        }

        JSONObject change = getChangeFromITipAnswer(iTipAnswer);
        ContentType calendarContentType = getCalendarContentType(session);
        if (iTipAnswer.has("messageType") && "cancel".equals(iTipAnswer.getString("messageType"))) {
            if (change != null && change.has("deletedAppointment")) {
                JSONObject app = change.getJSONObject("deletedAppointment");
                DataObject appObj = new SimpleDataObject(session, calendarContentType);
                fillAppObject(calendarContentType, app, appObj);
                appObj.setChangeState(ChangeState.DELETED);
                return appObj;
            } else if (change != null && change.has("currentAppointment")) {
                JSONObject app = change.getJSONObject("currentAppointment");
                DataObject appObj = new SimpleDataObject(session, calendarContentType);
                fillAppObject(calendarContentType, app, appObj);
                appObj.setChangeState(ChangeState.DELETED);
                return appObj;
            } else if (iTipAnswer.has("uid")) {
                DataObject appObj = new SimpleDataObject(session, calendarContentType);
                appObj.setFieldContent("uid", iTipAnswer.getString("uid"));
                appObj.setChangeState(ChangeState.DELETED);
                return appObj;
            }
        }
        if (change == null)
            return null;
        if (change.has("type") && "create".equals(change.getString("type"))) {
            JSONObject newApp = change.getJSONObject("newAppointment");
            DataObject newAppObj = new SimpleDataObject(session, calendarContentType);
            fillAppObject(calendarContentType, newApp, newAppObj);
            newAppObj.setFieldContent("created_by", String.valueOf(session.getUserIdentifier()));
            return newAppObj;
        }

        if (change.has("type") && "update".equals(change.getString("type"))) {
            JSONObject currApp = change.getJSONObject("currentAppointment");
            JSONObject newApp = change.getJSONObject("newAppointment");
            DataObject currAppObj = new SimpleDataObject(session, calendarContentType);
            DataObject newAppObj = new SimpleDataObject(session, calendarContentType);
            fillAppObject(calendarContentType, currApp, currAppObj);
            currAppObj.commitChanges();
            fillAppObject(calendarContentType, newApp, newAppObj);
            newAppObj.commitChanges();
            newAppObj = DataObjectUtil.copyAndModify(currAppObj, newAppObj, true, true);
            if (iTipAnswer.has("messageType") && "reply".equals(iTipAnswer.getString("messageType")))
                newAppObj.commitChanges(); // sets the state of the object to "unmodified" to be able to differentiate from meeting update
            return newAppObj;
        }
        return null;
    }

    public static void fillAppObject(ContentType calendarContentType, JSONObject newApp, DataObject newAppObj) throws JSONException {
        ContentTypeField[] fields = calendarContentType.getFields();
        for (ContentTypeField contentTypeField : fields) {
            String fieldName = contentTypeField.getFieldName();
            DataType<?> fieldType = contentTypeField.getFieldType();
            if (newApp.has(fieldName))
                newAppObj.setFieldContent(fieldName, fieldType.extractFromJSONObject(newAppObj.getSession(), newApp, fieldName));
        }
    }

    private static JSONObject getChangeFromITipAnswer(JSONObject iTipAnswer) throws JSONException {
        if (!iTipAnswer.has("changes"))
            return null;
        JSONArray changes = iTipAnswer.getJSONArray("changes");
        if (changes.length() == 0)
            return null;
        return changes.getJSONObject(0);
    }

    private static ContentType getCalendarContentType(Session session) throws StorageAccessException, USMStorageException {
        ContentType[] contentTypes = session.getContentTypes();
        ContentType calendarContentType = null;
        for (ContentType contentType : contentTypes) {
            if (contentType.getCode() == DefaultContentTypes.CALENDAR_CODE)
                calendarContentType = contentType;
        }
        return calendarContentType;
    }

}
