/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.datatypes.mail;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * This class realizes an object with a field for every non-standard header. 
 * The header name is the field name. The header value is the value of the field as string. 
 * @author ibr
 *
 */
public class Headers implements Serializable {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private final Map<String, String[]> _map;

	public Map<String, String[]> getMap() {
		return _map;
	}

	public Headers(Map<String, String[]> map) {
		if (map == null)
			_map = Collections.emptyMap();
		else
			_map = Collections.unmodifiableMap(map);
	}

	public Headers(JSONObject jSONObject) throws JSONException {
		if (jSONObject != null) {
			Map<String, String[]> map = new HashMap<String, String[]>();
			for (Iterator<?> iterator = jSONObject.keys(); iterator.hasNext();) {
				String key = (String) iterator.next();
				Object value = jSONObject.get(key);
				if (value instanceof String) {
					String[] strings = new String[1];
					strings[0] = value.toString();
					map.put(key, strings);
				} else if (value instanceof JSONArray) {
					JSONArray array = (JSONArray) value;
					String[] strings = new String[array.length()];
					for (int i = 0; i < strings.length; i++)
						strings[i] = array.isNull(i) ? "" : array.getString(i);
					map.put(key, strings);
				}

			}
			_map = Collections.unmodifiableMap(map);
		} else {
			_map = Collections.emptyMap();
		}
	}

	public Headers() {
		_map = Collections.emptyMap();
	}

	@Override
	public int hashCode() {
		return _map.hashCode();
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Headers other = (Headers) obj;

		//return _map.equals(other.getMap());
		return isMapEqual(other.getMap());
	}

	private boolean isMapEqual(Map<String, String[]> otherMap) {
		// TODO Improve the quality of the code in this method
		if (_map.size() != otherMap.size())
			return false;
		try {
			Iterator<Entry<String, String[]>> iterator = _map.entrySet().iterator();
			while (iterator.hasNext()) {
				Entry<String, String[]> entry = iterator.next();
				String key = entry.getKey();
				String[] element = entry.getValue();
				if (element == null) {
					if (!(otherMap.get(key) == null && otherMap.containsKey(key)))
						return false;
				} else {
					if (!Arrays.equals(element, otherMap.get(key)))
						return false;
				}
			}
		} catch (NullPointerException e1) {
			return false;
		}

		return true;
	}

	@Override
	public String toString() {
		if (_map == null)
			return "";
		StringBuilder sb = new StringBuilder(1000);
		for (String key : _map.keySet()) {
			if (sb.length() > 0)
				sb.append(';');
			String[] strings = _map.get(key);
			sb.append(key).append(':').append('[').append(strings[0]);
			if (strings.length > 1) {
				for (int i = 1; i < strings.length; i++) {
					sb.append(',').append(strings[i]);
				}
			}
			sb.append(']');
		}
		return sb.toString();
	}

	public JSONObject toJSONObject() throws JSONException {
		JSONObject result = new JSONObject();
		for (Map.Entry<String, String[]> entry : _map.entrySet()) {
			String key = entry.getKey();
			String[] data = entry.getValue();
			if (data == null || data.length == 0) {
				result.put(key, "");
			} else if (data.length == 1) {
				result.put(key, data[0]);
			} else {
				JSONArray array = new JSONArray();
				for (String s : data)
					array.put(s);
				result.put(key, array);
			}
		}
		return result;
	}
}
