/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.datatypes.contacts;

import java.io.Serializable;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.datatypes.AbstractDataType;
import com.openexchange.usm.api.session.Session;

public class ImageDataType extends AbstractDataType<Image> {

    public static final String IMAGE1 = "image1";
    
    private static final String IMAGE_LAST_MODIFIED = "image_last_modified";

    public static final String IMAGE1_URL = "image1_url";

    public static final String IMAGE1_CONTENT_TYPE = "image1_content_type";
    
    public ImageDataType() {
        super(IMAGE1);
    }

    @Override
    public Image checkValue(Object o) {
        if (o != null && !(o instanceof Image))
            generateTypeError(o);
        return (Image) o;
    }

    @Override
    public Image extractFromJSONArray(Session session, JSONArray array, int index) throws JSONException {
        // Special case, image consists of multiple columns: image_last_modified, image1_url and image1_content_type
        // String imageData = array.getString(index); // "image1"
        long timestamp = array.isNull(index) ? 0L : array.getLong(index); // "image_last_modified"
        if (timestamp == 0L)
            return null;
        String url = array.isNull(index + 1) ? null : array.getString(index + 1); // "image1_url"
        String contentType = array.isNull(index + 2) ? null : array.getString(index + 2); // "image1_content_type"
        return new Image(timestamp, url, contentType);
    }

    @Override
    public Image extractFromJSONObject(Session session, JSONObject jsonObject, String field) throws JSONException {
        // String imageData = jsonObject.getString(IMAGE1); // "image1"
        long timestamp = jsonObject.isNull(IMAGE_LAST_MODIFIED) ? 0L : jsonObject.getLong(IMAGE_LAST_MODIFIED);
        if (timestamp == 0L)
            return null;
        String url = jsonObject.isNull(IMAGE1_URL) ? null : jsonObject.getString(IMAGE1_URL);
        String contentType = jsonObject.isNull(IMAGE1_CONTENT_TYPE) ? null : jsonObject.getString(IMAGE1_CONTENT_TYPE);
        return new Image(timestamp, url, contentType);
    }

    @Override
    public void storeInJSONObject(Session session, JSONObject parent, String field, Object value) throws JSONException {
        Image image = (Image) value;
        if (image == null || image.getContentType() == null || image.getData() == null)
            parent.put(IMAGE1, JSONObject.NULL);
        // All other cases will be handled in special case in ContactContentTypeTransferHandler
    }

    @Override
    public void addToJSONArray(Session session, JSONArray parent, Object value) {
        // We store the lastModified timestamp in the JSONArray. This is mainly for toString() of SimpleDataObject, since the Contact Image
        // is not stored in an array.
        Image image = (Image) value;
        parent.put((image == null) ? 0L : image.getLastModified());
    }

    @Override
    public Serializable serialize(Object value) {
        return (value instanceof Image) ? ((Image) value).getLastModified() : null;
    }

    @Override
    public Image deserialize(Serializable value) {
        if (value instanceof Number)
            return new Image(((Number) value).longValue(), (String) null, (String) null);
        return (value instanceof Image) ? ((Image) value) : null;
    }
}
