/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.cache.ox_adapter;

import java.io.Serializable;
import java.util.Collections;
import java.util.Map;
import java.util.Properties;
import java.util.WeakHashMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.caching.CacheService;
import com.openexchange.exception.OXException;
import com.openexchange.usm.api.cache.Cache;
import com.openexchange.usm.api.cache.CacheManager;

/**
 * Cache Manager Adaptor. Uses the OX Caching service.
 * @author ldo
 *
 */
public class CacheManagerAdapter implements CacheManager {
    
    private static final Log LOG = LogFactory.getLog(CacheManagerAdapter.class);

	// Keep track of Caches in use for profiling / debugging purposes
	private final Map<Cache<?, ?>, String> _cacheMap = Collections.synchronizedMap(new WeakHashMap<Cache<?, ?>, String>());

	private CacheService _cacheService;

	/**
	 * Initializes a new {@link CacheManagerAdapter}.
	 * @param cacheService
	 */
	public CacheManagerAdapter(CacheService cacheService) {
		_cacheService = cacheService;
		LOG.info("USM->OX CacheManagerAdapter activated");
	}

	/**
	 * Deactivate
	 */
	public void deactivate() {
		for (String cacheId : _cacheMap.values()) {
			try {
				_cacheService.freeCache(cacheId);
			} catch (OXException e) {
				LOG.error("Can not free cache: " + cacheId, e);
			}
		}
		_cacheMap.clear();
		LOG.info("USM->OX CacheManagerAdapter deactivated");
	}

	@Override
    public <K extends Serializable, V extends Serializable> Cache<K, V> createCache(String id, Class<K> keyClass,
			Class<V> valueClass) {
		return createCache(id, keyClass, valueClass, null);
	}

	@Override
    public <K extends Serializable, V extends Serializable> Cache<K, V> createCache(String id, Class<K> keyClass,
			Class<V> valueClass, Properties properties) {
		com.openexchange.caching.Cache cacheImpl;
		try {
		    _cacheService.loadConfiguration(properties);
			cacheImpl = _cacheService.getCache(id);
		} catch (OXException e) {
			LOG.error("Can not create cache: " + id, e);
			return null; //TODO: maybe throw exception
		}
		CacheAdapter<K, V> cache = new CacheAdapter<K, V>(cacheImpl);
		_cacheMap.put(cache, id);
		if (LOG.isDebugEnabled())
			LOG.debug("Created Cache " + id + ", " + _cacheMap.size() + " Caches active");
		return cache;
	}
}
