/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.cache.ox_adapter;

import java.io.Serializable;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.exception.OXException;
import com.openexchange.usm.api.cache.Cache;

/**
 * Adapter for the Cache interface to the OX Caching Interface.
 *
 * @author ldo
 *
 * @param <K> Type of keys for Cache
 * @param <V> Type of values for Cache
 */
public class CacheAdapter<K extends Serializable, V extends Serializable> implements Cache<K, V> {

    private static final Log _journal = LogFactory.getLog(CacheAdapter.class);
    
	private final com.openexchange.caching.Cache _cacheImpl;
	
	/**
	 * Initializes a new {@link CacheAdapter}.
	 * @param cacheImpl
	 */
	public CacheAdapter(com.openexchange.caching.Cache cacheImpl) {
		_cacheImpl = cacheImpl;
	}

	@Override
    @SuppressWarnings("unchecked")
	public V get(K name) {
		return (V) _cacheImpl.get(name);
	}

	@Override
    public void put(K name, V obj, boolean invalidate) {
		try {
			_cacheImpl.put(name, obj, invalidate);
		} catch (OXException e) {
			_journal.error("Cache: put failed.", e);
		}
	}

	@Override
    public void putSafe(K name, V obj) {
		try {
			_cacheImpl.putSafe(name, obj);
		} catch (OXException e) {
			_journal.error("Cache: put safe failed.", e);
		}
	}

	@Override
    public void remove(K name) {
		try {
			_cacheImpl.remove(name);
		} catch (OXException e) {
			_journal.error("Cache: remove failed.", e);
		}
	}

	@Override
    public void dispose() {
		_cacheImpl.dispose();
	}
}
