/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.api.session.storage;

import java.util.List;
import java.util.Set;
import com.openexchange.usm.api.database.CachedDBEntry;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.Session;


/**
 * {@link PersistentSyncStateStorage}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public interface PersistentSyncStateStorage {
    /**
     * Stores the given {@link DataObjectSet} to the persistent storage.
     * 
     * @param session
     * @param objectID
     * @param element
     * @param oldTimestamp
     * @param newTimestamp
     * @param timestampToKeep
     * @return
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    public long store(Session session, String objectID, String element, long oldTimestamp, long newTimestamp, long timestampToKeep) throws StorageAccessException, USMStorageException;
    
    /**
     * Returns the cached data for the given objectID
     * @param objectID
     * @param maxSyncStatesInDB
     * @return {@link CachedData}
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    public List<CachedDBEntry> get(Session session, String objectID, int maxSyncStatesInDB) throws StorageAccessException, USMStorageException;
    
    /**
     * Removes all DB entries for the given object IDs.
     * 
     * @param session
     * @param objectIDs
     * @throws USMStorageException
     */
    public void remove(Session session, String... objectIDs) throws USMStorageException;
    
    /**
     * Returns a set of all object-ids for which data is stored and which are not in the provided set of object-ids.
     * 
     * @param session
     * @param excludedObjectIDs
     * @return a list with all objectIDs that have persistent storage but were not provided in the parameter set
     * @throws USMStorageException
     */
    public List<String> getRemainingObjectIDs(Session session, Set<String> excludedObjectIDs) throws USMStorageException;
    
    /**
     * Re-maps the already stored objectIDs.
     * 
     * @param session
     * @param oldObjectID
     * @param newObjectID
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    public void remapStates(Session session, String oldObjectID, String newObjectID) throws StorageAccessException, USMStorageException;
    
    /**
     * Removes all saved states for this session.
     * 
     * @param session
     * @throws USMStorageException
     */
    public void purgeSession(Session session) throws USMStorageException;
}
