/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.api.session.storage;

import java.util.List;
import java.util.Map;
import java.util.Set;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.assets.ContextSessionID;
import com.openexchange.usm.api.session.assets.SessionInformation;
import com.openexchange.usm.api.session.assets.USMSessionID;

/**
 * {@link PersistentSessionDataStorage}
 * 
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public interface PersistentSessionDataStorage {

    /**
     * Stores (inserts or updates) the field value for the given key. Any previously stored value is overwritten.
     * 
     * @param session
     * @param key
     * @param value
     * @throws USMStorageException
     * @throws StorageAccessException
     */
    public void storeFieldValue(Session session, String key, String value) throws USMStorageException, StorageAccessException;

    /**
     * Returns a map with all fields found for the given session
     * 
     * @param session
     * @return
     * @throws USMStorageException
     * @throws StorageAccessException
     */
    public Map<String, String> getFields(Session session) throws USMStorageException, StorageAccessException;

    /**
     * Removes the given field from the persistent storage
     * 
     * @param session
     * @param key field to be removed
     * @throws USMStorageException
     * @throws StorageAccessException
     */
    public void removeField(Session session, String key) throws USMStorageException, StorageAccessException;

    /**
     * Removes all fields for the given {@link Session}
     * 
     * @param session
     * @throws USMStorageException
     * @throws StorageAccessException
     */
    public void removeAllFieldsForSession(Session session) throws USMStorageException, StorageAccessException;

    /**
     * Checks whether or not a session exists in the persistent storage
     * 
     * @param session
     * @return
     * @throws USMStorageException
     * @throws StorageAccessException
     */
    public boolean sessionExists(Session session) throws USMStorageException, StorageAccessException;

    /**
     * Verify whether or not a session for a specific device exists in the persistent storage
     * 
     * @param session
     * @return
     * @throws USMStorageException
     * @throws StorageAccessException
     */
    public boolean sessionExistsForDevice(Session session) throws USMStorageException, StorageAccessException;

    /**
     * Inserts the given session to the persistent storage
     * 
     * @param session
     * @return the sessionID
     * @throws USMStorageException
     * @throws StorageAccessException
     */
    public int insertSession(Session session) throws USMStorageException, StorageAccessException;

    /**
     * Return the USMSessionID from the persistent storage, or <pre>null</pre> if the session do not exists.
     * @param session
     * @return
     * @throws USMStorageException
     * @throws StorageAccessException
     */
    public Integer getSessionID(Session session) throws USMStorageException, StorageAccessException;

    /**
     * Removes the given sessions from the persistent storage for the specified contextID, and returns the amount of the sessions that were
     * removed.
     * 
     * @param contextID
     * @param sessionIDs
     * @return the number of the removed sessions
     * @throws StorageAccessException
     */
    public int purgeSessions(int contextID, Set<ContextSessionID> sessionIDs) throws StorageAccessException;

    /**
     * Returns a set with all the sessionIDs that are to be removed.
     * 
     * @param contextID
     * @param ttl
     * @param validContextIDs
     * @return
     * @throws StorageAccessException
     */
    public Set<ContextSessionID> getSessionsToRemove(int contextID, long ttl, Set<Integer> validContextIDs) throws StorageAccessException;

    /**
     * Get the specified fields for the specified context from the the persistent storage.
     * 
     * @param contextForDB
     * @param cid
     * @param id
     * @param protocol
     * @param device
     * @param field
     * @param validContextIDs if not null and cid != 0: Specifies a set of valid contextIDs. Any data that is stored in the DB that does not
     *            belong to a context in the set is excluded
     * @return
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    public List<String> getFieldsFromSchema(int contextForDB, int cid, int id, String protocol, String device, String field, Set<Integer> validContextIDs) throws StorageAccessException, USMStorageException;

    /**
     * @param contextForDB
     * @param cid
     * @param id
     * @param protocol
     * @param device
     * @param field
     * @param value
     * @param validContextIDs if not null and cid != 0: Specifies a set of valid contextIDs. Any data that is stored in the DB that does not
     *            belong to a context in the set is excluded
     * @return
     * @throws USMStorageException
     * @throws StorageAccessException
     */
    public Map<USMSessionID, SessionInformation> updateFieldsInSchema(int contextForDB, int cid, int id, String protocol, String device, String field, String value, Set<Integer> validContextIDs) throws USMStorageException, StorageAccessException;

    /**
     * Returns all available context ids in the system (delegate to {@link DatabaseService})
     * 
     * @return
     * @throws StorageAccessException
     */
    public List<Integer> getAllContextIDs() throws StorageAccessException;

    /**
     * Returns all contexts that are in the same db schema (delegate to {@link DatabaseService})
     * 
     * @param contextID
     * @return
     * @throws StorageAccessException
     */
    public int[] getContextsInSameSchema(int contextID) throws StorageAccessException;
}
