/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.api.session.assets;

import java.io.Serializable;


/**
 * {@link SessionID} Identifier for an  USM session as seen from client/protocol side. A session is uniquely identified by user, protocol and device.
 *
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class SessionID implements Comparable<SessionID>, Serializable {

	private static final long serialVersionUID = 1L;

	private final String _user;
	private final String _protocol;
	private final String _device;
    private final int _hash;

	public SessionID(String user, String protocol, String device) {
	    super();
		_user = user;
		_protocol = protocol;
		_device = device;

		int prime = 31;
        int result = 1;
        result = prime * result + ((_device == null) ? 0 : _device.hashCode());
        result = prime * result + ((_protocol == null) ? 0 : _protocol.hashCode());
        result = prime * result + ((_user == null) ? 0 : _user.hashCode());
        _hash = result;
	}

	public String getUser() {
		return _user;
	}

	public String getProtocol() {
		return _protocol;
	}

	public String getDevice() {
		return _device;
	}

	@Override
    public int compareTo(SessionID o) {
		int result = _user.compareTo(o._user);
		if (result == 0) {
			result = _protocol.compareTo(o._protocol);
			if (result == 0) {
                result = _device.compareTo(o._device);
            }
		}
		return result;
	}

	@Override
	public String toString() {
		return _user + ':' + _protocol + ':' + _device;
	}

	@Override
	public int hashCode() {
		return _hash;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
            return true;
        }
		if (!(obj instanceof SessionID)) {
            return false;
        }
		SessionID other = (SessionID) obj;
		if (_user == null) {
            if (other._user != null) {
                return false;
            }
        } else if (!_user.equals(other._user)) {
            return false;
        }
		if (_protocol == null) {
            if (other._protocol != null) {
                return false;
            }
        } else if (!_protocol.equals(other._protocol)) {
            return false;
        }
		if (_device == null) {
			if (other._device != null) {
                return false;
            }
		} else if (!_device.equals(other._device)) {
            return false;
        }
		return true;
	}
}
