/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.api.session;

import java.io.Serializable;
import com.openexchange.usm.api.session.assets.ChangeState;

/**
 * Interface for a filter that can be applied to DataObjects, e.g. to filter out elements
 * that are on the server but the client protocol is not interested in them.
 * 
 * @author afe
 *
 */
public interface DataObjectFilter extends  Serializable {

	/**
	 * Called before a sync to allow initialization or resetting of any internal fields the filter may use to compute
	 * either  the acceptance of an object or if it should be added to the result.
	 */
	void initialize();

	/**
	 * Called with a DataObject. Return true if the DataObject should be included, false
	 * if it should be filtered out. The DataObjectFilter <b>must not</b> modify the
	 * DataObject in any way.
	 * 
	 * @param object
	 * @return
	 */
	boolean accept(DataObject object);

	/**
	 * Called for all operations that might be reported to the client (and are not already filtered out due to the simple
	 * limit parameter that limits the number of operations returned). If this method returns true, the operation will
	 * be reported and stored in the result, if it returns false, it will not be reported and not stored in the result. The
	 * result will be marked as incomplete.
	 * 
	 * @param action Operation that will be reported
	 * @param object current object state as stored in server
	 * @return
	 */
	boolean addOperationToResult(ChangeState action, DataObject object);
	
	/**
	 * This method can be used to provide an alternative sync-id that should be kept in the database instead of the
	 * one that is provided to the sync methods in Session. This can be useful if a protocol needs to perform any
	 * modifications to a stored sync state before the sync mechanism is called. That way the system still can ensure
	 * that the original sync id provided by a client is kept in the DB if the sync is unsuccessful.
	 * If this method returns 0, the normal mechanism is used.
	 * 
	 * @return
	 */
	long getOriginalSyncID();
}
