/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.api.contenttypes.mail;

import java.util.BitSet;
import java.util.List;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.resource.ResourceInputStream;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.io.InputStreamProvider;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Session;

/**
 * Handler for the entire interaction with the OX server for the Mail Content Type.
 * 
 * @author ldo
 */
public interface MailContentType extends ContentType {

    String FILTER_MARKED_AS_DELETED_MAILS = "filterMarkedAsDeletedMails";

    int ALL_FLAGS = 1 | 2 | 4 | 8 | 16 | 32 | 64 | 256;

    /**
     * Sends a mail as MIME data block.
     * 
     * @param mail
     * @param session
     * @throws AuthenticationFailedException
     * @throws OXCommunicationException
     */
    void sendMail(String mail, Session session) throws USMException;

    /**
     * Sends a mail as MIME data block from an InputStream.
     * 
     * @param streamProvider
     * @param session
     * @throws USMException
     * @throws AuthenticationFailedException
     * @throws OXCommunicationException
     */
    void sendMail(InputStreamProvider streamProvider, Session session) throws USMException;

    /**
     * Gets a mail data object from the server ready as reply or forward mail.
     * 
     * @param session
     * @param messageID
     * @param folderID
     * @param action
     * @param dropPrefix
     * @return
     * @throws USMException
     */
    DataObject getReplyForward(Session session, String messageID, String folderID, String action, String view, boolean dropPrefix) throws USMException;

    /**
     * Gets the mail source as byte array.
     * 
     * @param object
     * @param requestedFields
     * @return
     * @throws USMException
     */
    byte[] readMailSource(DataObject object, BitSet requestedFields) throws USMException;

    /**
     * Gets the mail source as byte array.
     * 
     * @param object
     * @param requestedFields
     * @param ignoreICalAttachments
     * @return
     * @throws USMException
     */
    byte[] readMailSource(DataObject object, BitSet requestedFields, boolean ignoreICalAttachments) throws USMException;

    /**
     * Creates a new mail and saves it in its parent folder. The mail is not sent out!
     * 
     * @param folderId
     * @param flag
     * @param mailMIMEdataBlock the MIME block as String (the entire email)
     * @param session
     * @return the ID of the newly created mail
     * @throws USMException
     */
    String createNewMail(String folderId, int flag, String mailMIMEdataBlock, Session session) throws USMException;

    /**
     * Creates a new mail from the provided InputStream and saves it in its parent folder. The mail is not sent out!
     * 
     * @param folderId
     * @param flags
     * @param streamProvider provider for InputStreams providing the MIME mail
     * @param session
     * @return the ID of the newly created mail
     * @throws USMException
     */
    String createNewMailFromStream(String folderId, int flags, InputStreamProvider streamProvider, Session session) throws USMException;

    /**
     * Retrieves the specified mail attachment for a mail as binary data.
     * 
     * @param session
     * @param folderId
     * @param mailId
     * @param attachmentId
     * @return the resource object
     * @throws USMException
     */
    Object getMailAttachment(Session session, String folderId, String mailId, String attachmentId) throws USMException;

    /**
     * Retrieves the specified mail attachment for a mail as binary data that can be read from the provided InputStream
     * 
     * @param session
     * @param folderId
     * @param mailId
     * @param attachmentId
     * @return the resource object
     * @throws USMException
     */
    ResourceInputStream getMailAttachmentAsStream(Session session, String folderId, String mailId, String attachmentId) throws USMException;

    /**
     * Retrieves the specified mail attachment for a mail as binary data that can be read from the provided InputStream. The attachment is provided as the ID that was previously returned by an OX getStructuredMail-call
     * 
     * @param session
     * @param folderId
     * @param mailId
     * @param structureId
     * @return the resource object
     * @throws USMException
     */
    ResourceInputStream getMailAttachmentStreamFromStructureId(Session session, String folderId, String mailId, String structureId) throws USMException;

    /**
     * Retrieves a mail from the server as structured mail object. The result is returned as JSONObject as it can not be stored in the
     * DataObject itself.
     * 
     * @param object the mail data object
     * @param maxAttachmentSize maximum size of attachments that will be directly reported
     * @return JSONObject
     * @throws USMException
     */
    JSONObject readStructuredMail(DataObject object, int maxAttachmentSize) throws USMException;

    /**
     * Converts an iCal attachment to a json appointment data object.
     * 
     * @param mailObject
     * @param attSeqId
     * @param timezone
     * @return
     * @throws USMException
     */
    JSONObject convertICalToJSON(DataObject mailObject, String attSeqId, String timezone) throws USMException;

    /**
     * Analyzes iCal attachment and proposes the adequate action.
     * 
     * @param mailObject
     * @param attSeqId
     * @return complex data object containing the proposed data changes and possible further actions.
     * @throws USMException
     */
    JSONObject analyzeICalAttachment(DataObject mailObject, String attSeqId) throws USMException;

    /**
     * Responds to a meeting request via iTIP on the server.
     * 
     * @param mailObject
     * @param attSeqId
     * @param confirmation
     * @return
     * @throws USMException
     */
    public JSONObject respondToMeetingRequest(DataObject mailObject, String attSeqId, int confirmation) throws USMException;

    /**
     * Updates appointment based on a meeting request mail. Uses the iTIP API on the server.
     * 
     * @param mailObject
     * @param attSeqId
     * @param action
     * @return
     * @throws USMException
     */
    public JSONObject updateObjectBasedOnMeetingRequest(DataObject mailObject, String attSeqId, String action) throws USMException;

    /**
     * Search mail folder on a given search criteria. (see
     * http://oxpedia.org/index.php?title=HTTP_API#Module_.22search.22_.28preliminary.29)
     * 
     * @param searchCriteria
     * @param folderId
     * @param session
     * @param requestedFields
     * @param firstMatch
     * @param lastMatch
     * @return
     */
    public List<DataObject> searchMails(JSONObject searchCriteria, String folderId, Session session, BitSet requestedFields, int firstMatch, int lastMatch) throws USMException;

    /**
     * Search mail folder for the given free text.
     * 
     * @param searchText
     * @param folderId
     * @param session
     * @param requestedFields
     * @param firstMatch
     * @param lastMatch
     * @return
     * @throws USMException
     */
    public List<DataObject> searchMails(String searchText, String folderId, Session session, BitSet requestedFields, int firstMatch, int lastMatch) throws USMException;

    /**
     * Gets the vacation rule for this session.
     * 
     * @param session
     * @return
     * @throws USMException
     */
    public JSONObject getVacationRule(Session session) throws USMException;

    /**
     * Creates new vacation rule.
     * 
     * @param session
     * @param rule
     * @throws USMException
     */
    public void setVacationRule(Session session, JSONObject rule) throws USMException;

    /**
     * Updates the vacation rule for this user.
     * 
     * @param session
     * @param rule
     * @throws USMException
     */
    public void updateVacationRule(Session session, JSONObject rule) throws USMException;

    /**
     * Deletes the vacation rule for this user.
     * 
     * @param session
     * @param rule
     * @throws USMException
     */
    public void deleteVacationRule(Session session, JSONObject rule) throws USMException;
    
    public boolean hasVacationRuleTimes(Session session) throws USMException;
}


