/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.api.cache;

import java.io.Serializable;
import java.util.Set;
import com.openexchange.usm.api.session.assets.CompleteSessionID;

/**
 * {@link SyncStateCache}: A cache for the sync states stored by a single USM session. A SyncStateCacheProvider may return the same
 * SyncStateCache for more than 1 USM session. The cache implementation has then to make sure that the stored data does not interfere
 * between sessions. Every method in this interface also provides the CompleteSessionID to identify the session. It is guaranteed that the
 * CompleteSessionID is always the same as the one that generated the cache (SyncStateCacheProvider.getCache()). The USM session is uniquely
 * identified by the SessionID within the CompleteSessionID, the cache may use the additional information (UserID) as it sees fit.
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public interface SyncStateCache {

    /**
     * Store the given sync state (data) for the given timestamp in the cache. A sync state is uniquely identified by the following tuple:
     * (SessionID, objectID, timestamp). Only store the data if no data already exists in the cache for the given tuple. Return true if
     * storage was successful, false otherwise.
     * 
     * @param session CompleteSessionID, same as when the SyncStateCache was retrieved from its SyncStateCacheProvider
     * @param objectID arbitrary object ID, normally not longer than 64 characters
     * @param timestamp syncid for the sync state
     * @param data sync state data
     * @return true if data was stored, false if not
     */
    boolean put(CompleteSessionID session, String objectID, long timestamp, Serializable[][] data);

    /**
     * Return a previously stored sync state from the cache. If no sync state is found for (SessionID, objectID, timestamp), return null.
     * 
     * @param session
     * @param objectID
     * @param timestamp
     * @return sync state for (SessionID, objectID, timestamp), or null if not found
     */
    Serializable[][] get(CompleteSessionID session, String objectID, long timestamp);

    /**
     * Remove the specified (obsolete) sync state from the cache (if found). It is no longer used by the sync system.
     * 
     * @param session
     * @param objectID
     * @param timestamp
     */
    void remove(CompleteSessionID session, String objectID, long timestamp);

    /**
     * Remove all sync states stored for all tuples (SessionID, objectID1),(SessionID, objectID2),... The objects have been removed (e.g.
     * folders have been deleted).
     * 
     * @param session
     * @param objectIDs
     */
    void removeSyncStates(CompleteSessionID session, String... objectIDs);

    /**
     * Remove all sync states stored except for all tuples (SessionID, objectID1),(SessionID, objectID2),... The given objects represent the
     * complete list of all synchronized folders, all other objects are no longer used.
     * 
     * @param session
     * @param objectIDs
     */
    void retainSyncStates(CompleteSessionID session, Set<String> objectIDs);

    /**
     * Change all stored sync states (i.e. for all timestamps) for the given tuple (SessionID, oldObjectID) by replacing oldObjectID with
     * newObjectID. Sometimes object IDs may change (e.g. rename or move of email folders), the stored sync states in the cache must then be
     * updated to reflect the new object ids.
     * 
     * @param session
     * @param oldObjectID
     * @param newObjectID
     */
    void remapStates(CompleteSessionID session, String oldObjectID, String newObjectID);
}
