/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.api.cache;

import java.io.Serializable;
import java.util.Properties;

/**
 * Provides cache(s) that can be used throughout the system. The actual
 * internal implementation can vary (e.g. a simple Map-mechanism could
 * be used, or something more complex like JCS).
 * 
 * @author afe
 *
 */
public interface CacheManager {
	/**
	 * Creates a cache for given key and value classes using a given ID. If
	 * properties is not specified (<code>null</code>), the cache manager will
	 * look in the USM configuration file for a property called com.openexchange.usm.cache.properties.&lt;id&gt;,
	 * where id equals the first parameter. If this property is set, is specifies the properties file to use
	 * for the cache. If it is not set, the manager will look for a property called
	 * com.openexchange.usm.cache.properties.default and use the properties file specified
	 * there. If this is also not available, the CacheManager will use default fallback properties.
	 * 
	 * @param <K> type that is valid for keys for the generated cache
	 * @param <V> type that is valid for values for the generated cache
	 * @param id identifier to use for the cache (for monitoring/logging purposes)
	 * @param keyClass Class that identifies the key type
	 * @param valueClass Class that identifies the value type
	 * @param properties additional properties to configure the cache 
	 *        if <code>null</code> than default cache properties are used
	 *        for each cache ID, the keys in the properties should start with: "jcs.region." + ID or "jcs.default" for using default properties
	 * @return Newly created cache
	 */
	<K extends Serializable, V extends Serializable> Cache<K, V> createCache(String id, Class<K> keyClass,
			Class<V> valueClass, Properties properties);

	/**
	 * Creates a cache for given key and value classes using a given ID. 
	 * Behaves as <code>createCache(id, keyClass, valueClass, null)</code>.
	 * 
	 * @param <K>
	 * @param <V>
	 * @param id
	 * @param keyClass
	 * @param valueClass
	 * @return
	 */
	<K extends Serializable, V extends Serializable> Cache<K, V> createCache(String id, Class<K> keyClass,
			Class<V> valueClass);
}
