/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.datatypes.tasks.calendar;

import java.io.Serializable;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Defines the User Participant Object used in the Tasks Module.
 * 
 * @author ldo
 */
public final class UserParticipantObject implements Serializable {

    private static final long serialVersionUID = 1L;

    public static final String CONFIRM_MESSAGE = "confirmmessage";

    public static final String CONFIRMATION = "confirmation"; /* 0=none, 1=accepted, 2=declined, 3=tentative */

    static final String DISPLAY_NAME = "display_name";

    static final String ID = "id";

    /** User ID */
    private final int _id;

    /** Displayable name of the participant */
    private final String _displayName;

    /** Valid values: 0-3 */
    private final int _confirmation;

    /** Confirm Message of the participant */
    private final String _confirmMessage;

    public UserParticipantObject(int id, String name, int confirmation, String message) {
        _id = id;
        _displayName = name;
        _confirmation = confirmation;
        _confirmMessage = message;
    }

    public UserParticipantObject(JSONObject jObject) throws JSONException {
        _id = jObject.getInt(ID);
        _displayName = jObject.has(DISPLAY_NAME) ? jObject.getString(DISPLAY_NAME) : "";
        _confirmation = jObject.hasAndNotNull(CONFIRMATION) ? jObject.getInt(CONFIRMATION) : 0;
        _confirmMessage = jObject.has(CONFIRM_MESSAGE) ? jObject.getString(CONFIRM_MESSAGE) : "";
    }

    public UserParticipantObject() {
        _id = 0;
        _displayName = "";
        _confirmation = 0;
        _confirmMessage = "";
    }

    public int getId() {
        return _id;
    }

    public String getDisplayName() {
        return _displayName;
    }

    public int getConfirmation() {
        return _confirmation;
    }

    public String getConfirmMessage() {
        return _confirmMessage;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + _id;
        result = prime * result + _displayName.hashCode();
        result = prime * result + _confirmation;
        result = prime * result + _confirmMessage.hashCode();
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        UserParticipantObject other = (UserParticipantObject) obj;
        if (_id != other.getId())
            return false;
        if (!_displayName.equals(other.getDisplayName()))
            return false;
        if (_confirmation != other.getConfirmation())
            return false;
        if (!_confirmMessage.equals(other.getConfirmMessage()))
            return false;
        return true;
    }

    @Override
    public String toString() {
        return '{' + ID + ':' + String.valueOf(_id) + ';' + DISPLAY_NAME + ':' + _displayName + ';' + CONFIRMATION + ':' + String.valueOf(_confirmation) + ';' + CONFIRM_MESSAGE + ':' + _confirmMessage + '}';
    }

    public JSONObject toJSONObject() throws JSONException {
        JSONObject result = new JSONObject();
        result.put(ID, _id);
        result.put(DISPLAY_NAME, _displayName);
        result.put(CONFIRMATION, _confirmation);
        if (!_confirmMessage.isEmpty()) {
            result.put(CONFIRM_MESSAGE, _confirmMessage);
        }
        return result;
    }
}
