/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.ox_json.impl;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpMethodBase;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.HttpURL;
import org.apache.commons.httpclient.HttpsURL;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.NameValuePair;
import org.apache.commons.httpclient.URI;
import org.apache.commons.httpclient.URIException;
import org.apache.commons.httpclient.methods.ByteArrayRequestEntity;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.PutMethod;
import org.apache.commons.httpclient.methods.RequestEntity;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.apache.commons.httpclient.methods.multipart.ByteArrayPartSource;
import org.apache.commons.httpclient.methods.multipart.FilePart;
import org.apache.commons.httpclient.methods.multipart.MultipartRequestEntity;
import org.apache.commons.httpclient.methods.multipart.Part;
import org.apache.commons.httpclient.methods.multipart.PartSource;
import org.apache.commons.httpclient.methods.multipart.StringPart;
import org.apache.commons.httpclient.params.HttpConnectionManagerParams;
import org.apache.commons.httpclient.params.HttpMethodParams;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.java.Streams;
import com.openexchange.usm.api.USMVersion;
import com.openexchange.usm.api.contenttypes.resource.ResourceInputStream;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.ConflictingChangeException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.TemporaryDownOrBusyException;
import com.openexchange.usm.api.exceptions.USMIllegalStateException;
import com.openexchange.usm.api.exceptions.USMStartupException;
import com.openexchange.usm.api.io.InputStreamProvider;
import com.openexchange.usm.api.ox.json.JSONResult;
import com.openexchange.usm.api.ox.json.JSONResultType;
import com.openexchange.usm.api.ox.json.OXJSONAccess;
import com.openexchange.usm.api.ox.json.OXJSONPropertyNames;
import com.openexchange.usm.api.ox.json.OXResource;
import com.openexchange.usm.api.session.OXConnectionInformation;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.util.OXJSONUtil;

/**
 * @author afe
 */
public class OXJSONAccessImpl implements OXJSONAccess {

    private static final String OX_REQUEST_INFORMATION_FIELD = "OXJSONAccessImpl.OXRequestInformation";

    private static final String CUSTOM_HOST_HEADER = "X-Host";

    private static final int MAX_RETRY = 10;

    private static final int MAX_REDIRECT_COUNT = 10;

    private static final String IMAGE = "image/";

    private static final String TEXT_HTML = "text/html";

    private static final String APPLICATION_OCTET_STREAM = "application/octet-stream";

    private static final String APPLICATION_OCTET_STREAM_UTF_8 = "application/octet-stream;charset=UTF-8";

    private final static boolean ALWAYS_COLLECT_STATISTICS = true;

    private final static int RESOURCE_BUFFER_DEFAULT_SIZE = 40000;

    private final static String FILE_PART_NAME = "file";

    private final static String STRING_PART_NAME = "json";

    private final static String BINARY_ENCODING = "binary";

    private final static String TEXT_JAVASCRIPT_CONTENTTYPE = "text/javascript";

    private final static String CHARSET_UTF8 = "UTF-8";

    private final static String CONFIG = "config/";

    private final static String DATA = "data";

    private final static String HTTPS_PREFIX = "https:";

    // Check after 5 minutes if redirect still active, restricts number of log messages and redirect tests
    private static final long REDIRECT_RETEST_INTERVAL = 300000L;

    private static final long LOCK_TIMEOUT = 100;

    private static final long _STREAM_INACTIVITY_TIMEOUT = 20000L;

    private static final String USM_USER_AGENT = "Open-Xchange USM HTTP Client";

    private static final Log LOG = LogFactory.getLog(OXJSONAccessImpl.class);

    private MultiThreadedHttpConnectionManager _httpManager;

    private Thread _backgroundConnectionCloser;

    private int _connectionTimeout;

    private String _oxJSONBaseURLasString;

    private HttpURL _oxJSONBaseURL;

    private long _lastRedirect = 0L;

    private boolean _passClientHostHeader = false;

    /**
     * Initializes a new {@link OXJSONAccessImpl}.
     *
     * @param oxJSONURL
     * @param maxConnections
     * @param connectionTimeout
     * @param performStaleChecking
     * @param passHostHeaderFromClient
     */
    public OXJSONAccessImpl(String oxJSONURL, int maxConnections, int connectionTimeout, boolean performStaleChecking, boolean passHostHeaderFromClient, String reloginErrorCodes, String missingObjectErrorCodes) {
        String error = doActivate(oxJSONURL, maxConnections, connectionTimeout, performStaleChecking, passHostHeaderFromClient, reloginErrorCodes, missingObjectErrorCodes);
        if (error != null) {
            error = "OXJSONAccess not activated, reason: " + error;
            LOG.error(error);
            deactivate(null);
            throw new USMStartupException(OXJSONErrorCode.ACCESSIMPL_OXAJAXACCESS_NOTACTIVATED_ENCODING_NUMBER_8, error);
        }
    }

    @Override
    public void login(Session session) throws AuthenticationFailedException, OXCommunicationException {
        logout(session, false);
        try {
            OXConnectionData result = obtainConnectionData(session);
            if (LOG.isDebugEnabled()) {
                LOG.debug(session + " Logged in, session id = " + result.getSessionID());
            }
        } catch (AuthenticationFailedException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug(session + " Authentication failed", e);
            }
            throw e;
        } catch (OXCommunicationException e) {
            LOG.warn(session + " Error while authenticating", e);
            throw e;
        }
    }

    @Override
    public JSONObject getConfiguration(Session session, String... path) throws AuthenticationFailedException, OXCommunicationException {
        JSONObject config = readConfiguration(session, path);
        if (LOG.isDebugEnabled()) {
            LOG.debug(session + " Retrieved OX ajax configuration");
        }
        return config;
    }

    private JSONObject readConfiguration(Session session, String... path) throws AuthenticationFailedException, OXCommunicationException {
        String accessPath = CONFIG;
        if (path.length > 0) {
            StringBuilder sb = new StringBuilder(CONFIG);
            for (String part : path) {
                sb.append(part).append('/');
            }
            accessPath = sb.toString();
        }
        JSONResult result = doGet(accessPath, null, session, null);
        if (result.getResultType() != JSONResultType.JSONObject) {
            throw new OXCommunicationException(
                OXJSONErrorCode.ACCESSIMPL_ERROR_WHILE_AUTHENTICATING_NUMBER_3,
                "Unknown OX response reading configuration",
                result.toString());
        }
        try {
            Object object = result.getJSONObject().get(DATA);
            if (object instanceof String) {
                return new JSONObject().put(DATA, object);
            } else if (object instanceof JSONObject) {
                return (JSONObject) object;
            }
            if (object instanceof JSONArray) {
                return new JSONObject().put(DATA, object);
            } else if (object instanceof Integer) {
                return new JSONObject().put(DATA, String.valueOf(object));
            } else {
                throw new OXCommunicationException(
                    OXJSONErrorCode.ACCESSIMPL_ILLEGAL_DATA_BY_OXCONFIG_RESPONSE_NUMBER_4,
                    "OX configuration response contains illegal data",
                    result.toString());
            }
        } catch (JSONException e) {
            throw new OXCommunicationException(
                OXJSONErrorCode.ACCESSIMPL_ILLEGAL_DATA_BY_OXCONFIG_RESPONSE_NUMBER_4,
                "OX configuration response contains illegal data",
                e,
                result.toString());
        }
    }

    private OXConnectionData performLogin(Session session) throws AuthenticationFailedException, OXCommunicationException {
        String user = session.getUser();
        String password = session.getPassword();
        HttpClient client = new HttpClient(_httpManager);
        client.getParams().setParameter("http.protocol.content-charset", "UTF-8");
        client.getParams().setParameter(HttpMethodParams.USER_AGENT, USM_USER_AGENT);
        PostMethod method = new PostMethod(
            getBaseURL() + "login?client=" + urlEncode("USM-" + session.getProtocol()) + "&version=" + urlEncode(USMVersion.VERSION_BUILD) + "&authId=" + UUID.randomUUID() + createClientIPParameter(session));
        method.addParameter("action", "login");
        method.addParameter("name", user);
        method.addParameter("password", password);
        for (String header : session.getXRequestHeaders().keySet()) {
            if (CUSTOM_HOST_HEADER.equals(header) && _passClientHostHeader) {
                method.setRequestHeader(header, session.getXRequestHeaders().get(header));
            } else {
                method.setRequestHeader(header, session.getXRequestHeaders().get(header));
            }
        }
        JSONResult result = executeMethodOnce(session, client, method);
        if (result.getResultType() == JSONResultType.Error) {
            throw new AuthenticationFailedException(
                OXJSONErrorCode.ACCESSIMPL_AUTHENTICATION_FAILED_NUMBER_5,
                "Authentication failed",
                result.getJSONObject());
        } else if (result.getResultType() != JSONResultType.JSONObject) {
            throw new OXCommunicationException(
                OXJSONErrorCode.ACCESSIMPL_INVALID_RESULT_FROM_SERVER,
                "Server responce does not contain JSONObject",
                result.toString());
        }
        try {
            String sessionID = result.getJSONObject().getString("session");
            int userId = result.getJSONObject().getInt("user_id");
            int contextId = result.getJSONObject().getInt("context_id");
            OXConnectionData data = new OXConnectionData(sessionID, userId, contextId, client);
            session.setOXConnectionData(data);
            return data;
        } catch (JSONException e) {
            throw new AuthenticationFailedException(
                OXJSONErrorCode.ACCESSIMPL_RESPONSE_NOT_CONTAIN_SESSIONID_NUMBER_6,
                "Server response did not contain session ID",
                result.getJSONObject());
        }
    }

    private static String createClientIPParameter(Session session) {
        return ("127.0.0.1".equals(session.getClientIp()) ? "" : ("&clientIP=" + session.getClientIp()));
    }

    @Override
    public void logout(Session session) {
        logout(session, true);
    }

    private void logout(Session session, boolean doLog) {
        if (doLog && LOG.isDebugEnabled()) {
            LOG.debug(session + " Logging out");
        }
        if (!obtainOptionalLock(session, "login?action=logout")) {
            LOG.warn("Timeout while waiting to log out session " + session);
            session.setOXConnectionData(null);
            return;
        }
        try {
            OXConnectionInformation info = session.getOXConnectionData();
            session.setOXConnectionData(null);
            performLogout(info);
        } finally {
            releaseLock(session);
        }
    }

    @Override
    public void logout(OXConnectionInformation info) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Logging out " + info);
        }
        performLogout(info);
    }

    private void performLogout(OXConnectionInformation info) {
        if (!(info instanceof OXConnectionData)) {
            return;
        }
        OXConnectionData data = (OXConnectionData) info;
        GetMethod method = new GetMethod(getBaseURL() + "login?action=logout&session=" + urlEncode(data.getSessionID()));
        try {
            // Request statistics can not be collected here, no request may be associated, USM session may no longer be in memory
            executeMethod(data.getHttpClient(), method);
        } catch (HttpException ignored) {
            // do nothing on failed logout
        } catch (IOException ignored) {
            // do nothing on failed logout
        } finally {
            method.releaseConnection();
        }
    }

    @Override
    public JSONResult doGet(String path, String action, Session session, Map<String, String> parameters) throws AuthenticationFailedException, OXCommunicationException {
        OXConnectionData data = obtainConnectionData(session);
        String url = getBaseURL() + path;
        if (LOG.isDebugEnabled()) {
            LOG.debug(session + " GET " + url + " action " +  action);
            if (parameters != null && !parameters.isEmpty())
            	LOG.debug(session + "parameters " + parameters.toString());
        }
        GetMethod method = new GetMethod(url);
        method.setQueryString(generateQueryParameters(action, data.getSessionID(), parameters));
        return executeMethodAndExtractResult(session, data, method, action, parameters);
    }

    @Override
    public JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, JSONArray array) throws AuthenticationFailedException, OXCommunicationException {
        return doPut(path, action, session, parameters, array.toString());
    }

    @Override
    public JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, JSONObject object) throws AuthenticationFailedException, OXCommunicationException {
        return doPut(path, action, session, parameters, object.toString());
    }

    @Override
    public JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, String str) throws AuthenticationFailedException, OXCommunicationException {
        OXConnectionData data = obtainConnectionData(session);
        String url = getBaseURL() + path;
        if (LOG.isDebugEnabled()) {
            LOG.debug(session + " PUT " + url + '\n' + str);
        }
        PutMethod method = new PutMethod(url);
        method.setQueryString(generateQueryParameters(action, data.getSessionID(), parameters));
        try {
            method.setRequestEntity(new StringRequestEntity(str, TEXT_JAVASCRIPT_CONTENTTYPE, CHARSET_UTF8));
        } catch (UnsupportedEncodingException e1) {
            throw new USMIllegalStateException(
                OXJSONErrorCode.ACCESSIMPL_UNSUPPORTED_ENCODING_NUMBER_7,
                "UnsupportedEncoding: Character encoding UTF-8 not found");
        }
        return executeMethodAndExtractResult(session, data, method, action, parameters);
    }

    @Override
    public JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, byte[] bytes) throws AuthenticationFailedException, OXCommunicationException {
        return doPut(path, action, session, parameters, "byte array", new ByteArrayRequestEntity(bytes));
    }

    @Override
    public JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, InputStreamProvider streamProvider) throws AuthenticationFailedException, OXCommunicationException {
        return doPut(path, action, session, parameters, "InputStream", new InputStreamProviderRequestEntity(streamProvider));
    }

    private JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, String type, RequestEntity entity) throws AuthenticationFailedException, OXCommunicationException {
        OXConnectionData data = obtainConnectionData(session);
        String url = getBaseURL() + path;
        if (LOG.isDebugEnabled()) {
            LOG.debug(session + " PUT " + url + " with " + type);
        }
        PutMethod method = new PutMethod(url);
        method.setQueryString(generateQueryParameters(action, data.getSessionID(), parameters));
        method.setRequestEntity(entity);
        return executeMethodAndExtractResult(session, data, method, action, parameters);
    }

    public void deactivate() {
        deactivate("USM OXJSONAccess deactivated");
    }

    private String doActivate(String oxJSONURL, int maxConnections, int connectionTimeout, boolean performStaleChecking, boolean passHostHeaderFromClient, String reloginErrorCodes, String missingObjectErrorCodes) {
        _oxJSONBaseURLasString = oxJSONURL;
        _passClientHostHeader = passHostHeaderFromClient;
        try {
            _oxJSONBaseURL = getURLFromString(oxJSONURL);
        } catch (URIException e) {
            return "OXJSONAccess base URL is no valid http(s) URL: " + _oxJSONBaseURLasString;
        }
        OXJSONUtil.setReloginErrorCodes(reloginErrorCodes);
        OXJSONUtil.setMissingObjectErrorCodes(missingObjectErrorCodes);
        _connectionTimeout = connectionTimeout;
        _httpManager = new MultiThreadedHttpConnectionManager();
        HttpConnectionManagerParams httpParams = _httpManager.getParams();
        httpParams.setDefaultMaxConnectionsPerHost(maxConnections);
        httpParams.setMaxTotalConnections(maxConnections);
        httpParams.setConnectionTimeout(_connectionTimeout);
        httpParams.setStaleCheckingEnabled(performStaleChecking);
        _backgroundConnectionCloser = new Thread("USM-OX-HttpConnectionCloser") {

            @Override
            public void run() {
                try {
                    while (!isInterrupted()) {
                        Thread.sleep(getConnectionTimeout());
                        getHttpConnectionManager().closeIdleConnections(getConnectionTimeout());
                    }
                } catch (InterruptedException ignored) {
                    // We get interrupted to indicate that this thread has to stop, so simply stop
                }
            }
        };
        _backgroundConnectionCloser.setPriority(Thread.MIN_PRIORITY);
        _backgroundConnectionCloser.setDaemon(true);
        _backgroundConnectionCloser.start();
        LOG.info("USM OXJSONAccess activated, OX-URL=" + _oxJSONBaseURLasString);
        return null;
    }

    private static HttpURL getURLFromString(String urlString) throws URIException {
        return (urlString.toLowerCase().startsWith(HTTPS_PREFIX)) ? new HttpsURL(urlString, CHARSET_UTF8) : new HttpURL(
            urlString,
            CHARSET_UTF8);
    }

    private void deactivate(String message) {
        if (_backgroundConnectionCloser != null && _backgroundConnectionCloser.isAlive()) {
            _backgroundConnectionCloser.interrupt();
        }
        _backgroundConnectionCloser = null;
        _oxJSONBaseURL = null;
        _oxJSONBaseURLasString = null;
        if (message != null) {
            LOG.info(message);
        }
    }

    private OXConnectionData obtainConnectionData(Session session) throws AuthenticationFailedException, OXCommunicationException {
        obtainRequiredLock(session, "<obtain OX connection data>");
        try {
            OXConnectionData o = getCurrentOXConnectionData(session);
            LOG.debug(session + " Retrieving OX session with ID: " + o);
            return (o == null) ? performLogin(session) : o;
        } finally {
            releaseLock(session);
        }
    }

    private static OXConnectionData getCurrentOXConnectionData(Session session) {
        OXConnectionInformation o = session.getOXConnectionData();
        return (o instanceof OXConnectionData) ? ((OXConnectionData) o) : null;
    }

    private static NameValuePair[] generateQueryParameters(String action, String sessionID, Map<String, String> params) {
        List<NameValuePair> list = new ArrayList<NameValuePair>();
        if (action != null) {
            list.add(new NameValuePair("action", action));
        }
        if (sessionID != null) {
            list.add(new NameValuePair("session", sessionID));
        }
        if (params != null) {
            for (Map.Entry<String, String> entry : params.entrySet()) {
                list.add(new NameValuePair(entry.getKey(), entry.getValue()));
            }
        }
        list.add(new NameValuePair("timezone", "UTC"));
        return list.toArray(new NameValuePair[list.size()]);
    }

    private String getBaseURL() {
        return _oxJSONBaseURLasString;
    }

    private static String urlEncode(String s) {
        try {
            return URLEncoder.encode(s, CHARSET_UTF8);
        } catch (UnsupportedEncodingException e) {
            throw new USMIllegalStateException(
                OXJSONErrorCode.ACCESSIMPL_UTF8_ENCODING_NOTFOUND_NUMBER_10,
                "Character encoding UTF-8 not found");
        }
    }

    private JSONResult executeMethodAndExtractResult(Session session, OXConnectionData data, HttpMethodBase method, String action, Map<String, String> parameters) throws AuthenticationFailedException, OXCommunicationException {
        obtainRequiredLock(session, method.getName() + ' ' + method.getPath() + '?' + method.getQueryString());
        try {
            JSONResult result = executeMethodOnce(session, data.getHttpClient(), method);
            // On specific error for expired/unknown session, do login and repeat
            if (OXJSONSupport.shouldPerformNewLogin(result)) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug(session + " Session with sessionid " + data.getSessionID() + " unknown to OX server, performing login and repeat");
                }
                login(session);
                data = obtainConnectionData(session);
                method.setQueryString(generateQueryParameters(action, data.getSessionID(), parameters));
                setRequestInformation(session, method.getName() + ' ' + method.getPath() + '?' + method.getQueryString());
                result = executeMethodOnce(session, data.getHttpClient(), method);
            }
            for (int i = 0; i < MAX_RETRY && shouldRetryRequest(result); i++) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Server returned error from category: " + OXJSONUtil.getJSONErrorCategory(result.getJSONObject()) + ". Retrying request..");
                }
                result = executeMethodOnce(session, data.getHttpClient(), method);
            }
            if (shouldRetryRequest(result)) {
                throw new TemporaryDownOrBusyException(
                    OXJSONErrorCode.ACCESSIMPL_TOO_MANY_DOWN_OR_BUSY_RETRIES,
                    "Server/Resource is temporary down or busy.",
                    result.getJSONObject());
            }
            return result;
        } finally {
            releaseLock(session);
        }
    }

    private JSONResult executeMethodOnce(Session session, HttpClient client, HttpMethodBase method) throws OXCommunicationException {
        boolean collectOXStatistics = ALWAYS_COLLECT_STATISTICS || LOG.isDebugEnabled();
        long startTime = collectOXStatistics ? System.currentTimeMillis() : 0L;
        try {
            int statusCode = executeMethod(client, method);
            if (statusCode != HttpStatus.SC_OK) {
                if (LOG.isDebugEnabled()) {
                    StringBuffer responseHeaders = new StringBuffer();
                    for (Header header: method.getResponseHeaders()) {
                        responseHeaders.append("  " + header.getName() + ":" + header.getValue() + "\n");
                    }
                    LOG.debug("OX communication failed, response headers:\n" + responseHeaders.toString());
                }
                if (statusCode == 429) {
                    Header header = method.getResponseHeader("Retry-After");
                    if (header != null) {
                        Header[] headers = { header };
                        throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_HTTP_CONNECTION_ERROR_11, statusCode, headers);
                    }
                }
                throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_HTTP_CONNECTION_ERROR_11, statusCode);
            }
            JSONResult result = OXJSONSupport.parseResult(extractResponseBodyAsBufferedReader(method), null);
            if (isSynchronizationConflictError(result)) {
                throw new ConflictingChangeException(
                    OXJSONErrorCode.ACCESSIMPL_CONFLICT_CHANGE_ON_OXACCESS_NUMBER_12,
                    "Conflicting change on OX access",
                    result.getJSONObject());
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("RESULT: " + result);
            }
            return result;
        } catch (HttpException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_HTTPEXCEPTION_OCCURED_NUMBER_13, e);
        } catch (IOException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_EXECUTE_METHOD_ERROR_NUMBER_14, e);
        } finally {
            method.releaseConnection();
            if (collectOXStatistics) {
                collectOXStatistics(session, startTime);
            }
        }
    }

    public static void collectOXStatistics(Session session, long startTime) {
        long duration = System.currentTimeMillis() - startTime;
        long number = getSessionProperty(session, OXJSONPropertyNames.NUMBER_OF_CALLS);
        session.setCustomProperty(OXJSONPropertyNames.NUMBER_OF_CALLS, Long.valueOf(number + 1L));
        long accTime = getSessionProperty(session, OXJSONPropertyNames.ACCUMULATED_TIME);
        session.setCustomProperty(OXJSONPropertyNames.ACCUMULATED_TIME, Long.valueOf(accTime + duration));
    }

    private static long getSessionProperty(Session session, String name) {
        Object o = session.getCustomProperty(name);
        return (o instanceof Long) ? ((Long) o).longValue() : 0L;
    }

    private static boolean isSynchronizationConflictError(JSONResult result) {
        return result.getResultType() == JSONResultType.Error && OXJSONUtil.ERROR_CODE_OBJECT_CHANGED.equals(OXJSONUtil.getJSONErrorCode(result.getJSONObject()));
    }

    private static boolean shouldRetryRequest(JSONResult result) {
        if (result.getResultType() != JSONResultType.Error) {
            return false;
        }
        if (OXJSONErrorCode.MSG_0043.equals(OXJSONUtil.getJSONErrorCode(result.getJSONObject()))) {
            return false;
        }
        String errorCategory = OXJSONUtil.getJSONErrorCategory(result.getJSONObject());
        return OXJSONUtil.CATEGORY_RESOURCE_TEMPORARY_DOWN.equals(errorCategory)
        // || OXJSONSupport.CATEGORY_RESOURCE_FULL_OR_BUSY.equals(errorCategory)
        || OXJSONUtil.CATEGORY_SOCKET_CONNECTION_CORRUPT.equals(errorCategory) || OXJSONUtil.CATEGORY_SUBSYSTEM_DOWN.equals(errorCategory);
    }

    private ResourceInputStreamImpl accessResource(Session session, String path, Map<String, String> parameters) throws AuthenticationFailedException, OXCommunicationException {
        GetMethod method = null;
        boolean freeMethod = true;
        try {
            String resourceMessage = session + " GET " + _oxJSONBaseURLasString + " + " + path;
            if (LOG.isDebugEnabled()) {
                LOG.debug(resourceMessage);
            }
            OXConnectionData connectionData = obtainConnectionData(session);
            boolean collectOXStatistics = ALWAYS_COLLECT_STATISTICS || LOG.isDebugEnabled();
            long startTime = collectOXStatistics ? System.currentTimeMillis() : 0L;
            if (parameters == null) {
                HttpURL url = getURLFromString(path);
                if (url.isRelativeURI()) {
                    url = (_oxJSONBaseURL instanceof HttpsURL) ? new HttpsURL((HttpsURL) _oxJSONBaseURL, path) : new HttpURL(
                        _oxJSONBaseURL,
                        url);
                }
                method = new GetMethod();
                String query = url.getQuery();
                query = query + "&session=" + connectionData.getSessionID();
                url.setQuery(query);
                method.setURI(url);
            } else {
                String url = getBaseURL() + path;
                method = new GetMethod(url);
                method.setQueryString(generateQueryParameters(null, connectionData.getSessionID(), parameters));
            }
            int statusCode = executeMethod(connectionData.getHttpClient(), method);
            if (statusCode != HttpStatus.SC_OK) {
                throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_INVALID_STATUSCODE_NUMBER_15, statusCode);
            }
            if (LOG.isTraceEnabled()) {
                StringBuffer headersList = new StringBuffer();
                headersList.append("Response-Headers:\n");
                for (Header header : method.getResponseHeaders()) {
                    headersList.append("  " + header.getName() + ":" + header.getValue() + "\n");
                }
                LOG.debug(headersList.toString());
            }
            Header responseHeader = method.getResponseHeader("Content-Type");
            if (responseHeader == null) {
                throw new OXCommunicationException(
                    OXJSONErrorCode.ACCESSIMPL_NO_CONTENT_TYPE_RECEIVED,
                    "No content type in response header for path: " + method.getURI().getPathQuery());
            }
            String contentType = responseHeader.getValue().toLowerCase();
            if (isValidResourceContentType(contentType)) {
                ResourceInputStreamImpl ris = new ResourceInputStreamImpl(this, session, connectionData, method, contentType, resourceMessage, startTime);
                freeMethod = false;
                return ris;
            }
            if (contentType == null || !contentType.startsWith(TEXT_HTML)) {
                throw new OXCommunicationException(
                    OXJSONErrorCode.ACCESSIMPL_INVALID_CONTENT_TYPE_RECEIVED,
                    "Invalid content type in response: " + contentType);
            }
            String httpResult = extractResponseBodyAsString(method);
            int startIndex = httpResult.indexOf("({");
            int endIndex = httpResult.lastIndexOf("})");
            JSONResult json = new JSONErrorResult(new JSONObject());
            if (startIndex > 0 && endIndex > startIndex) {
                json = checkPostResult(httpResult.substring(startIndex + 1, endIndex + 1));
            }
            throw new OXCommunicationException(
                OXJSONErrorCode.ACCESSIMPL_ERROR_ON_READING_RESOURCE,
                "Error on getting resource from Server",
                json.getJSONObject());
        } catch (URIException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_INVALID_URL_NUMBER_16, "Invalid URL specified: " + path, e);
        } catch (HttpException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_GETRESOURCE_ERROR_NUMBER_17, e);
        } catch (IOException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_GETRESOURCE_ERROR_NUMBER_18, e);
        } finally {
            if (freeMethod && method != null) {
                method.releaseConnection();
            }
        }
    }

    @Override
    public OXResource getResource(Session session, String path, Map<String, String> parameters) throws AuthenticationFailedException, OXCommunicationException {
        obtainRequiredLock(session, "GET " + path + ' ' + parameters);
        ResourceInputStreamImpl data = null;
        try {
            data = accessResource(session, path, parameters);
            try {
                return new OXResource(data.getContentType(), extractResponseBody(data.getMethod()));
            } finally {
                data.close();
            }
        } catch (IOException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_CAN_NOT_READ_RESOURCE, e);
        } finally {
            if(data == null) {
                releaseLock(session);
            }
        }
    }

    @Override
    public ResourceInputStream getResourceAsStream(Session session, String path, Map<String, String> parameters) throws AuthenticationFailedException, OXCommunicationException {
        obtainRequiredLock(session, "(as stream) GET " + path + ' ' + parameters);
        try {
            return accessResource(session, path, parameters);
        } catch (AuthenticationFailedException e) {
            releaseLock(session);
            throw e;
        } catch (OXCommunicationException e) {
            releaseLock(session);
            throw e;
        }
    }

    @Override
    public JSONResult storeResource(String path, String action, Session session, Map<String, String> parameters, JSONObject jsonData, byte[] imageByteData, String fileContentType) throws AuthenticationFailedException, OXCommunicationException {
        return storeResource(path, action, session, parameters, jsonData, imageByteData, fileContentType, STRING_PART_NAME, FILE_PART_NAME);
    }

    @Override
    public JSONResult storeResourceFromStream(String path, String action, Session session, Map<String, String> parameters, JSONObject jsonData, String fileContentType, final long size, final InputStreamProvider streamProvider) throws AuthenticationFailedException, OXCommunicationException {
        return storeResourceFromStream(
            path,
            action,
            session,
            parameters,
            jsonData,
            fileContentType,
            STRING_PART_NAME,
            FILE_PART_NAME,
            size,
            streamProvider);
    }

    @Override
    public JSONResult storeResource(String path, String action, Session session, Map<String, String> parameters, JSONObject jsonData, byte[] imageByteData, String fileContentType, String jsonPartName, String filePartName) throws AuthenticationFailedException, OXCommunicationException {
        ByteArrayPartSource source = new ByteArrayPartSource(filePartName, imageByteData);
        return storeResource(path, action, session, parameters, jsonData, fileContentType, jsonPartName, filePartName, source);
    }

    @Override
    public JSONResult storeResourceFromStream(String path, String action, Session session, Map<String, String> parameters, JSONObject jsonData, String fileContentType, String jsonPartName, final String filePartName, final long size, final InputStreamProvider streamProvider) throws AuthenticationFailedException, OXCommunicationException {
        PartSource source = new PartSource() {

            @Override
            public long getLength() {
                return size;
            }

            @Override
            public String getFileName() {
                return filePartName;
            }

            @Override
            public InputStream createInputStream() throws IOException {
                return streamProvider.getInputStream();
            }
        };
        return storeResource(path, action, session, parameters, jsonData, fileContentType, jsonPartName, filePartName, source);
    }

    private JSONResult storeResource(String path, String action, Session session, Map<String, String> parameters, JSONObject jsonData, String fileContentType, String jsonPartName, String filePartName, PartSource dataSource) throws AuthenticationFailedException, OXCommunicationException {
        PostMethod multiPartPostMethod = null;
        boolean collectOXStatistics = ALWAYS_COLLECT_STATISTICS || LOG.isDebugEnabled();
        long startTime = 0L;
        try {
            OXConnectionData connectionData = obtainConnectionData(session);
            String url = getBaseURL() + path;

            if (LOG.isDebugEnabled()) {
                LOG.debug(session + " POST " + url);
            }

            multiPartPostMethod = new PostMethod(url);
            multiPartPostMethod.setQueryString(generateQueryParameters(action, connectionData.getSessionID(), parameters));

            // "text/javascript", "UTF-8"
            List<Part> parts = new ArrayList<Part>();

            if (jsonData != null) {
                StringPart jsonPart = new StringPart(jsonPartName, jsonData.toString(), CHARSET_UTF8);
                jsonPart.setContentType(TEXT_JAVASCRIPT_CONTENTTYPE);
                parts.add(jsonPart);
            }

            FilePart filePart = new FilePart(filePartName, dataSource, fileContentType, BINARY_ENCODING);
            filePart.setTransferEncoding(BINARY_ENCODING);
            parts.add(filePart);

            multiPartPostMethod.setRequestEntity(new MultipartRequestEntity(
                parts.toArray(new Part[parts.size()]),
                multiPartPostMethod.getParams()));

            if(collectOXStatistics) {
                startTime = System.currentTimeMillis();
            }
            int statusCode = executeMethod(connectionData.getHttpClient(), multiPartPostMethod);
            if (statusCode != HttpStatus.SC_OK) {
                throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_INVALID_STATUSCODE_NUMBER_19, statusCode);
            }
            String httpResult = extractResponseBodyAsString(multiPartPostMethod);
            return checkAndExtractResult(httpResult);
        } catch (URIException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_INVALID_URL_NUMBER_20, "Invalid URL specified", e);
        } catch (HttpException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_STORE_RESOURCE_ERROR_NUMBER_21, e);
        } catch (IOException e) {
            throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_STORE_RESOURCE_ERROR_NUMBER_22, e);
        } finally {
            if (multiPartPostMethod != null) {
                multiPartPostMethod.releaseConnection();
            }
            if(collectOXStatistics && startTime != 0L) {
                collectOXStatistics(session, startTime);
            }
        }
    }

    private static JSONResult checkAndExtractResult(String httpResult) throws OXCommunicationException {
        httpResult = httpResult.trim();
        int startIndex = httpResult.indexOf("({");
        int endIndex = httpResult.lastIndexOf("})");
        if (startIndex > 0 && endIndex > startIndex) {
            String json = httpResult.substring(startIndex + 1, endIndex + 1);
            return checkPostResult(json);
        }
        throw new OXCommunicationException(
            OXJSONErrorCode.ACCESSIMPL_INVALID_RESULTOF_POST_REQUEST_NUMBER_23,
            "Result of POST request doesn't match configured pattern",
            httpResult);
    }

    private static JSONResult checkPostResult(String json) throws OXCommunicationException {
        try {
            return OXJSONSupport.generateJSONResultForJSONObject(new JSONObject(json), JSONResultType.JSONObject);
        } catch (JSONException e) {
            throw new OXCommunicationException(
                OXJSONErrorCode.ACCESSIMPL_RESULTOF_POST_NOTCONTAIN_VALID_JSONOBJECT_NUMBER_24,
                "Result of POST request doesn't contain valid JSONObject: " + json,
                e);
        }
    }

    private static byte[] extractResponseBody(HttpMethod method) throws IOException {
        InputStream in = null;
        try {
            in = method.getResponseBodyAsStream();

            ByteArrayOutputStream sink = Streams.newByteArrayOutputStream(2048);
            byte[] buf = new byte[512];
            for (int read; (read = in.read(buf, 0, buf.length)) > 0;) {
                sink.write(buf, 0, read);
            }
            Streams.close(in);
            in = null;
            return sink.toByteArray();
        } finally {
            Streams.close(in);
        }
    }

    private static BufferedReader extractResponseBodyAsBufferedReader(HttpMethodBase method) throws IOException {
        return new BufferedReader(new InputStreamReader(method.getResponseBodyAsStream(), method.getResponseCharSet()));
    }

    private static String extractResponseBodyAsString(HttpMethodBase method) throws IOException {
        InputStream in = null;
        InputStreamReader reader = null;
        try {
            in = method.getResponseBodyAsStream();
            reader = new InputStreamReader(in, method.getResponseCharSet());

            StringBuilder sb = new StringBuilder(2048);
            char[] cbuf = new char[512];
            for (int read; (read = reader.read(cbuf, 0, cbuf.length)) > 0;) {
                sb.append(cbuf, 0, read);
            }
            Streams.close(reader, in);
            reader = null;
            in = null;
            return sb.toString();
        } finally {
            Streams.close(reader, in);
        }
    }

    private int executeMethod(HttpClient client, HttpMethod method) throws HttpException, IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(method.getName() + method.getPath() + "&" + method.getQueryString());
            StringBuffer requestHeaders = new StringBuffer();
            for (Header header: method.getRequestHeaders()) {
                requestHeaders.append("  " + header.getName() + ":" + header.getValue() + "\n");
            }
            LOG.debug("Headers:\n" + requestHeaders.toString());
        }
        
        int status = HttpStatus.SC_NOT_FOUND;
        method.setFollowRedirects(false);
        String redirectFromServer = null;
        // override potentially set secure mode for secret cookie (when com.openexchange.forceHTTPS=true)
        for (org.apache.commons.httpclient.Cookie c : client.getState().getCookies()) {
            if (c.getName().startsWith("open-xchange-secret")) {
                c.setSecure(false);
            }
        }
        for (int i = 0; i < MAX_REDIRECT_COUNT; i++) {
            status = client.executeMethod(method);
            Header location = getRedirection(method, status);
            if (location == null) {
                if (i > 0 && _lastRedirect < System.currentTimeMillis() - REDIRECT_RETEST_INTERVAL) {
                    _lastRedirect = System.currentTimeMillis();
                    LOG.info("HTTP Access to JSON interface of OX-Server was redirected. Redirect from server: '" + redirectFromServer + "', used URL: '" + method.getURI() + "'");
                }
                return status;
            }
            redirectFromServer = location.getValue();
            URI newURI = new URI(redirectFromServer, true, method.getParams().getUriCharset());
            if (newURI.isRelativeURI()) {
                newURI = new URI(method.getURI(), newURI);
            }
            method.setURI(newURI);
        }
        LOG.error("Too many HTTP redirects accessing JSON interface of OX-Server");
        return status;
    }

    private static Header getRedirection(HttpMethod method, int status) {
        return (status == HttpStatus.SC_MOVED_PERMANENTLY || status == HttpStatus.SC_SEE_OTHER || status == HttpStatus.SC_TEMPORARY_REDIRECT) ? method.getResponseHeader("location") : null;
    }

    @Override
    public JSONResult doPost(String path, String action, Session session, Map<String, String> parameters) throws AuthenticationFailedException, OXCommunicationException {

        OXConnectionData data = obtainConnectionData(session);
        PostMethod method = new PostMethod(getBaseURL() + path);
        method.setQueryString(generateQueryParameters(action, data.getSessionID(), parameters));

        JSONResult result = executeMethodOnce(session, data.getHttpClient(), method);
        if (result.getResultType() == JSONResultType.Error) {
            throw new AuthenticationFailedException(
                OXJSONErrorCode.ACCESSIMPL_AUTHENTICATION_FAILED,
                "Authentication failed",
                result.getJSONObject());
        } else if (result.getResultType() != JSONResultType.JSONObject) {
            throw new OXCommunicationException(
                OXJSONErrorCode.ACCESSIMPL_INVALID_RESULT_FROM_SERVER_2,
                "Server responce does not contain JSONObject",
                result.toString());
        }
        return result;
    }

    private boolean obtainOptionalLock(Session session, String requestInformation) {
        try {
            return tryLock(session, requestInformation);
        } catch (InterruptedException e) {
            return false;
        }
    }

    private void obtainRequiredLock(Session session, String requestInformation) throws OXCommunicationException {
        try {
            if (tryLock(session, requestInformation)) {
                return;
            }
        } catch (InterruptedException e) {
            throw new OXCommunicationException(
                OXJSONErrorCode.INTERRUPTED_WHILE_WAITING_FOR_ACCESS,
                "Interrupted while waiting for exclusive access to OX server, locked by " + session + '/' + getRequestInformation(session),
                e);
        }
        throw new OXCommunicationException(
            OXJSONErrorCode.FAILED_TO_GAIN_EXCLUSIVE_OX_ACCESS,
            "Timeout while waiting for exclusive access to OX server, locked by " + session + '/' + getRequestInformation(session),
            (String) null);
    }

    private boolean tryLock(Session session, String requestInformation) throws InterruptedException {
        for (int i = 0; i < LOCK_TIMEOUT; i += 10) {
            if (tryLock(session, requestInformation, 10)) {
                return true;
            }
        }
        return false;
    }

    private boolean tryLock(Session session, String requestInformation, long timeout) throws InterruptedException {
        OXConnectionData data = getCurrentOXConnectionData(session);
        if (data != null) {
            ResourceInputStreamImpl currentRIS = data.getCurrentResourceInputStream();
            if (currentRIS != null && currentRIS.getLastActivity() < System.currentTimeMillis() - _STREAM_INACTIVITY_TIMEOUT) {
                LOG.info("Closing ResourceInputStream due to long inactivity (was locked by " + getRequestInformation(session) + "): " + currentRIS);
                try {
                    currentRIS.close();
                } catch (IOException ignored) {
                    // do not log error on auto-close, other request that took too long will log an error due to the closed InputStream
                }
            }
        }
        boolean success = session.getOXConnectionLock().tryLock(timeout, TimeUnit.SECONDS);
        if (success) {
            setRequestInformation(session, requestInformation);
        }
        return success;
    }

    public void releaseLock(Session session) {
        // Do not clear request information, keep data until next request to have it available for concurrency issues
        // setRequestInformation(session, null);
        try {
            // Probably not necessary, just to be sure to avoid a deadlock in case one of the access methods fails to cleanup properly
            session.getOXConnectionLock().unlock();
        } catch (IllegalMonitorStateException ignored) {
            // If the lock was acquired in another thread, the unlock() call will fail.
            // This can only happen if OX was accessed using one of the streaming methods.
            // Since in that case we additionally close the ResourceInputStream, that other
            // thread will get notified and release the lock itself.
        }
    }

    private static void setRequestInformation(Session session, String requestInformation) {
        session.setCustomProperty(OX_REQUEST_INFORMATION_FIELD, requestInformation);
    }

    private static Object getRequestInformation(Session session) {
        return session.getCustomProperty(OX_REQUEST_INFORMATION_FIELD);
    }

    private static boolean isValidResourceContentType(String contentType) {
        return (contentType != null) && (APPLICATION_OCTET_STREAM.equals(contentType) || APPLICATION_OCTET_STREAM_UTF_8.equalsIgnoreCase(contentType) || contentType.startsWith(IMAGE));
    }

    @Override
    public MultiThreadedHttpConnectionManager getHttpConnectionManager() {
        return _httpManager;
    }

    public int getConnectionTimeout() {
        return _connectionTimeout;
    }

    public MultiThreadedHttpConnectionManager getHttpManager() {
        return _httpManager;
    }
}
