/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.session.sync;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.datatypes.DataType;
import com.openexchange.usm.api.exceptions.ConflictingChange;
import com.openexchange.usm.api.exceptions.ConflictingChangeException;
import com.openexchange.usm.api.exceptions.InternalUSMException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.DataObjectFilter;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.assets.ChangeState;
import com.openexchange.usm.api.session.assets.ConflictResolution;
import com.openexchange.usm.api.session.assets.SyncResult;
import com.openexchange.usm.api.session.exceptions.SlowSyncRequiredException;
import com.openexchange.usm.api.session.exceptions.SynchronizationConflictException;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.dataobject.DataObjectUtil;
import com.openexchange.usm.session.dataobject.FieldMapStorageImpl;
import com.openexchange.usm.session.impl.SessionManagerImpl;
import com.openexchange.usm.session.impl.USMSessionManagementErrorCodes;

/**
 * 
 * @author afe
 *
 */
public class IncrementalContentSyncer extends BaseContentSyncer {
	private final static boolean MARK_CHANGE_FAILED_ON_SIMULTANEOUS_DELETION = true;

	public IncrementalContentSyncer(SessionManagerImpl sessionManager, int retryCount) {
		super(sessionManager, retryCount);
	}

	public SyncResult syncChangesWithServer(ContentSyncerStorage storage, ConflictResolution conflictResolution,
			DataObject[] lastSyncState, DataObject[] dataObjectChanges, long timestamp, int resultLimit,
			DataObjectFilter filter, Comparator<? super DataObject> sorter) throws USMException {
	    
	    if (_journal.isTraceEnabled()) {
	        String lastSyncObjects = "";
	        String clientChanges = "";
	        if (lastSyncState != null) {
	            for (DataObject f : lastSyncState) {
	                if (f.getChangeState() != ChangeState.UNMODIFIED)
	                    lastSyncObjects += "presync: lastSyncState " + f.toString() + "\n"; 
	            }
	        }
            if (dataObjectChanges != null) {
                for (DataObject f : dataObjectChanges) {
                    if (f.getChangeState() != ChangeState.UNMODIFIED)
                        clientChanges += "presync: clientChange " + f.toString() + "\n"; 
                }
            }
	    
            if (!lastSyncObjects.isEmpty() || !clientChanges.isEmpty())
                _journal.trace(lastSyncObjects + clientChanges);
	    }


	    SyncResult result = syncChangesWithServerCore(storage, conflictResolution,
            lastSyncState, dataObjectChanges, timestamp, resultLimit, filter, sorter);


	    if (_journal.isTraceEnabled()) {
            String newSyncObjects = "";
            String changes = "";
            DataObject[] newSyncState = result.getNewState();
            DataObject[] changedObjects = result.getChanges();
            if (newSyncState != null) {
                for (DataObject f : newSyncState) {
                    if (f.getChangeState() != ChangeState.UNMODIFIED)
                        newSyncObjects += "postsync: newSyncState " + f.toString() + "\n"; 
                }
            }
            if (changedObjects != null) {
                for (DataObject f : changedObjects) {
                    if (f.getChangeState() != ChangeState.UNMODIFIED)
                        changes += "postsync: serverChange " + f.toString() + "\n"; 
                }
            }
        
            if (!newSyncObjects.isEmpty() || !changes.isEmpty())
                _journal.trace(newSyncObjects + changes);
        }
	    
	    return result;
	}

	    
    private SyncResult syncChangesWithServerCore(ContentSyncerStorage storage, ConflictResolution conflictResolution,
            DataObject[] lastSyncState, DataObject[] dataObjectChanges, long timestamp, int resultLimit,
            DataObjectFilter filter, Comparator<? super DataObject> sorter) throws USMException {
		if (lastSyncState == null)
			throw new SlowSyncRequiredException(USMSessionManagementErrorCodes.NO_CACHED_DATA_INCR_SYNC,
					"No cached DataObjects for client available");
		//set the parent folder owner id field in all client data objects
		setParentFolderOwnerIDField(dataObjectChanges);
		//	1) retrieve cached information on DataObjects
		//		- if data not found in cache, generate error
		//	2) check if client operations are valid on cached DataObjects
		//		- if not valid, generate error
		DataObjectSet clientChangeSet = new DataObjectSet(dataObjectChanges);
		DataObjectSet cachedDataObjects = new DataObjectSet(lastSyncState);
		for (DataObject f : dataObjectChanges) {
			checkRequestedChanges(cachedDataObjects, f);
		}
		if (_journal.isDebugEnabled())
			_journal.debug(storage + " Starting incremental sync, resolution=" + conflictResolution + ", timestamp="
					+ timestamp + ',' + dataObjectChanges.length + " client requests");
		for (int count = 0; count <= _retryCount; count++) {
			if (filter != null)
				filter.initialize();
			DataObjectSet newestCachedData = null;
			for (DataObject f : dataObjectChanges)
				f.setFailed(false);
			//	3) retrieve server information on DataObjects
			DataObjectSet clientDataObjects = new DataObjectSet(cachedDataObjects);
			DataObjectSet serverDataObjects = getCurrentOrCachedServerData(storage, dataObjectChanges.length == 0);
			List<DataObject> serverCreations = serverDataObjects.toList();
			//	4) determine differences between server and cache data
			DataObjectSet serverModifications = new DataObjectSet();
			List<DataObject> clientResults = new ArrayList<DataObject>();
			for (Iterator<DataObject> i = serverCreations.iterator(); i.hasNext();) {
				DataObject sf = i.next();
				DataObject cf = clientDataObjects.get(sf.getID());
				if(cf == null)
				    cf = storage.findMatchingClientObjectForServerObject(clientDataObjects, sf);
				if (cf == null) {
					// Lookup UUID for all DataObjects that are new since the last sync state
					newestCachedData = _sessionManager.insertStoredUUID(sf, cachedDataObjects, newestCachedData);
					UUID serverUUID = sf.getUUID();
					if (serverUUID == null) {
						serverUUID = UUID.randomUUID();
						sf.setUUID(serverUUID);
					}
					cf = clientChangeSet.get(sf.getUUID());
					if (cf != null) {
						cf = cf.createCopy(true);
						cf.commitChanges();
					}
				}
				if (cf != null) {
					// if modified, store modification
					if (!sf.equals(cf)) {
                        DataObject serverModify = DataObjectUtil.copyAndModify(cf, sf, true);
						serverModifications.add(serverModify);
						if (_journal.isTraceEnabled()) {
						    _journal.trace("syncChangesWithServer copyAndModify: " + serverModify.toString());
						}
                    }
					sf.linkUUIDTo(cf);
					i.remove(); // found -> remove from list of server creations
					clientDataObjects.remove(cf); // and from temporary list of old objects which will become the list of deletions
				}
			}
			DataObjectSet serverDeletions = clientDataObjects;
			List<ConflictingChange> conflicts = new ArrayList<ConflictingChange>();
			List<DataObject> clientRequests = new ArrayList<DataObject>();
			// Create list of relevant client requests, i.e. only those for DataObjects that have not been deleted on the server
			DataObjectSet serverObjectsWithUUID = new DataObjectSet(serverDataObjects);
			for (DataObject f : dataObjectChanges) {
			    if (_journal.isTraceEnabled()) {
			        _journal.trace("syncChangesWithServer clientChange: " + f.toString());
			    }
				if (!isDuplicateCreation(serverObjectsWithUUID, f, serverCreations, cachedDataObjects)) {
					DataObject serverDeletion = serverDeletions.get(f.getID());
					if (serverDeletion == null)
						clientRequests.add(f);
					else if (f.getChangeState() != ChangeState.DELETED) {
						if (conflictResolution == ConflictResolution.ERROR
								|| conflictResolution == ConflictResolution.USE_CLIENT) {
							serverDeletion = serverDeletion.createCopy(true);
							serverDeletion.setChangeState(ChangeState.DELETED);
							conflicts.add(new ConflictingChange(f, serverDeletion));
						} else if (MARK_CHANGE_FAILED_ON_SIMULTANEOUS_DELETION) {
							f.setFailed(true);
						}
					}
				}
			}
			//	5) compare server with client operations
			//		- build list of changes to send to server
			//		- build list of changes to report to client
			//		- determine if conflicts exist
			//	6) resolve conflicts (move to list of changes to server or to client or generate error)
			List<DataObject> newServerDeletions = new ArrayList<DataObject>();
			List<DataObject> newServerCreations = new ArrayList<DataObject>();
			List<DataObject> newServerModifications = new ArrayList<DataObject>();
			// Check each client request for possible conflict with server changes
			for (DataObject f : clientRequests) {
				checkPossibleConflicts(conflictResolution, serverModifications, conflicts, newServerDeletions,
						newServerCreations, newServerModifications, f);
			}
			if (!conflicts.isEmpty())
				throw new SynchronizationConflictException(USMSessionManagementErrorCodes.CONFLICTS_INCR_SYNC,
						"Conflicting changes between server and client", conflicts
								.toArray(new ConflictingChange[conflicts.size()]));
			Map<DataObject, USMException> errorMap = new HashMap<DataObject, USMException>();
			//	7) send all updates to server
			//		- if sync error occurs, retry from step 3
			List<DataObject> serverChanges = new ArrayList<DataObject>();
			try {
				ContentSyncerSupport.updateOnServer(_sessionManager, serverDataObjects, newServerCreations,
						newServerModifications, newServerDeletions, errorMap, serverChanges);
				//serverModifications.addAll(serverChanges);
			} catch (ConflictingChangeException sce) {
				_journal.warn(storage + " Conflicting change while performing incremental sync", sce);
				continue;
			}

			boolean incomplete = false;
			Set<Object> objectGroups = new HashSet<Object>();

			for (DataObject f : sortElements(serverDeletions, sorter)) {
				DataObject cf = clientChangeSet.get(f.getID());
				if (cf == null || cf.getChangeState() != ChangeState.DELETED) {
					// If we have no limit or haven't reached it, add deletion to client result, otherwise add original to new synchronized state
					if (shouldBeAddedToResult(resultLimit, clientResults.size(), filter, ChangeState.DELETED, f,
							objectGroups)) {
						DataObject deletion = f.createCopy(true);
						deletion.setChangeState(ChangeState.DELETED);
						clientResults.add(deletion);
					} else {
						incomplete = true;
						serverDataObjects.add(f);
					}
				}
			}

			// Only add as many server modifications to the result as the limit dictates, undo the remaining server modifications in sync state
			for (DataObject modification : sortElements(serverModifications, sorter)) {
				if (shouldBeAddedToResult(resultLimit, clientResults.size(), filter, ChangeState.MODIFIED,
						modification, objectGroups)) {
					clientResults.add(modification);
				} else {
					incomplete = true;
					// add replaces here since modification and original object in cache have the same id
					String modificationId = modification.getID();
					DataObject cachedObj = cachedDataObjects.get(modificationId);
					if (cachedObj == null) {
						cachedObj = clientChangeSet.get(modificationId);
						if (cachedObj != null)
							cachedObj = cachedObj.createCopy(false);
						//happens when a client creation is modified by the server
					}
					if (cachedObj != null) {
						serverDataObjects.add(cachedObj);
					} else {
						serverDataObjects.remove(modificationId);
					}
				}
			}

			//Add server modifications - only as much as the limit allows
			for (DataObject modification : sortElements(serverChanges, sorter)) {
				if (shouldBeAddedToResult(resultLimit, clientResults.size(), filter, ChangeState.MODIFIED,
						modification, objectGroups)) {
					clientResults.add(modification);
					DataObject toRemove = serverDataObjects.get(modification.getID());
					if (toRemove != null) {
						serverDataObjects.remove(toRemove);
						DataObject modCopy = modification.createCopy(true);
						modCopy.commitChanges();
						serverDataObjects.add(modCopy);
					}
				} else {
					incomplete = true;
					//the server changes should not be removed from the data object list because they are still not in the list
				}
			}

			// Only add as many server creations to the result as the limit dictates, undo the remaining server creations in sync state
			for (DataObject sf : sortElements(serverCreations, sorter)) {
				if (shouldBeAddedToResult(resultLimit, clientResults.size(), filter, ChangeState.CREATED, sf,
						objectGroups)) {
					sf = sf.createCopy(true);
					sf.setChangeState(ChangeState.CREATED);
					clientResults.add(sf);
				} else {
					incomplete = true;
					serverDataObjects.remove(sf.getID());
				}
			}

			if (_oxDataCache != null)
				_oxDataCache.setSyncComplete(!incomplete);
			//	8) write DataObjects in cache
			//	9) report changes to client
			SyncResult result = ContentSyncerSupport.storeDataObjectsAndBuildClientResult(incomplete, storage,
					serverDataObjects, clientResults, errorMap, dataObjectChanges);
			if (_journal.isDebugEnabled())
				_journal.debug(storage + " Incremental sync finished: " + result.getDescription());
			return result;
		}
		throw new SynchronizationConflictException(USMSessionManagementErrorCodes.TOO_MANY_CONFICTS_INCR_SYNC,
				"Too many repeated conflicts on incremental sync");
	}

    private static boolean isDuplicateCreation(DataObjectSet serverCreationSet, DataObject f, List<DataObject> serverCreations, DataObjectSet cachedDataobjects) {
        if (f.getChangeState() == ChangeState.CREATED && f.getUUID() != null) {
            DataObject sc = serverCreationSet.get(f.getUUID());
            if (sc == null && DefaultContentTypes.FOLDER_ID.equals(f.getContentType().getID()) && DefaultContentTypes.MAIL_ID.equals(((Folder) f).getElementsContentTypeID())) {
                Object t = f.getFieldContent("title");
                String title = (t instanceof String) ? (String)t : null;
                String parentFolderID = f.getParentFolderID();
                if(DataObjectUtil.findMailFolder(parentFolderID, title, cachedDataobjects) != null)
                    return false; // not a duplicate creation, attempt to create folder which already exist on server
                sc = DataObjectUtil.findMailFolder(parentFolderID, title, serverCreationSet);
            }
            if (sc != null) {
                // If a creation with a UUID already exists in the server, nothing is to do, since the creation was already performed in a
                // previous call, but the result did not reach the client
                f.setID(sc.getID());
                serverCreations.remove(sc);
                return true;
            }
        }
        return false;
    }

	private void checkRequestedChanges(DataObjectSet cachedDataObjects, DataObject f) throws InternalUSMException {
		switch (f.getChangeState()) {
			case CREATED:
				if (cachedDataObjects.get(f.getUUID()) == null)
					return;
				throw new IllegalArgumentException("Invalid change requested: " + f);
			case UNMODIFIED:
			case DELETED:
			case MODIFIED:
				DataObject sf = cachedDataObjects.get(f.getID());
				if (sf != null)
					return;
				sf = cachedDataObjects.get(f.getUUID());
				if (sf != null) {
					f.setID(sf.getID());
					return;
				}
				throw new IllegalArgumentException("Invalid change requested: " + f);
		}
		throw new InternalUSMException(USMSessionManagementErrorCodes.ILLEGAL_CHANGE_STATE_NUMBER3,
				"Illegal ChangeState " + f.getChangeState());
	}

	private void checkPossibleConflicts(ConflictResolution conflictResolution, DataObjectSet serverModifications,
			List<ConflictingChange> conflicts, List<DataObject> newServerDeletions,
			List<DataObject> newServerCreations, List<DataObject> newServerModifications, DataObject f)
			throws InternalUSMException {
		switch (f.getChangeState()) {
			case UNMODIFIED:
				return;
			case CREATED:
				newServerCreations.add(f);
				// TODO Should we check if there is a matching DataObject created by the server ?
				return;
			case DELETED:
				DataObject serverChange = serverModifications.get(f.getID());
				if (serverChange != null) {
					if (conflictResolution == ConflictResolution.USE_SERVER) {
						f.setFailed(true);
						return;
					} else if (conflictResolution == ConflictResolution.ERROR) {
						conflicts.add(new ConflictingChange(f, serverChange));
						return;
					}
					serverModifications.remove(serverChange);
				}
				newServerDeletions.add(f);
				return;
			case MODIFIED:
				serverChange = serverModifications.get(f.getID());
				if (serverChange == null || isClientChangeWithServerCompatible(f, serverChange)) {
				    if (_journal.isTraceEnabled())
		                _journal.trace("serverModificationByTheClient: " + f.toString());
					newServerModifications.add(f);
					return;
				}
				handleConflictingModifications(conflictResolution, serverModifications, conflicts,
						newServerModifications, f, serverChange);
				return;
		}
		throw new InternalUSMException(USMSessionManagementErrorCodes.ILLEGAL_CHANGE_STATE_NUMBER4,
				"Illegal ChangeState " + f.getChangeState());
	}

	// check whether modified fields on the server are already contained in the client change
	private boolean isClientChangeWithServerCompatible(DataObject f, DataObject serverChange) {
        ContentTypeField[] fields = f.getContentType().getFields();
        Integer lastModifiedIndex = FieldMapStorageImpl.getInstance().getFieldNameToIndexMap(f.getContentType()).get("last_modified");
        if (lastModifiedIndex == null)
            lastModifiedIndex = -1;
        Integer noteIndex = FieldMapStorageImpl.getInstance().getFieldNameToIndexMap(f.getContentType()).get("note");
        if (noteIndex == null)
            noteIndex = -1;

        for (int i = 0; i < fields.length; i++) {
            if (i == lastModifiedIndex)
                continue;
            DataType<?> type = fields[i].getFieldType();
            if (serverChange.isFieldModified(i)) {
                if (!type.isEqual(f.getFieldContent(i), serverChange.getFieldContent(i))) {
                    if  (i == noteIndex) {
                        // we manipulate in AbstractSyncDelegate.simulateServerChangesForAppointments() the note with irrelevant changes 
                        String clientNote = f.getFieldContent(i) == null ? "" : f.getFieldContent(i).toString().trim();
                        String serverNote = serverChange.getFieldContent(i) == null ? "" : serverChange.getFieldContent(i).toString().trim();
                        if (clientNote.equals(serverNote)) {
                            continue;
                        }
                    }
                    if (_journal.isTraceEnabled()) {
                        String clientField = f.getFieldContent(i) == null ? "null" : f.getFieldContent(i).toString();
                        String serverField = serverChange.getFieldContent(i) == null ? "null" : serverChange.getFieldContent(i).toString();
                        _journal.trace("client-server-conflict of field " + fields[i].toString() + ": " + clientField + " != " + serverField);
                    }
                    return false;
                }
            }
        }

	    return true;
	}
	
	private void handleConflictingModifications(ConflictResolution conflictResolution,
			DataObjectSet serverModifications, List<ConflictingChange> conflicts,
			List<DataObject> newServerModifications, DataObject f, DataObject serverChange) throws InternalUSMException {
        if (_journal.isTraceEnabled()) {
            _journal.trace("client-server-conflict for: " + f.toString());
        }
		switch (conflictResolution) {
			case ERROR:
			case ERROR_DELETE_OVER_CHANGE:
				// TODO Mark client request as failed instead of throwing an exception ?
				conflicts.add(new ConflictingChange(f, serverChange));
				return;
			case USE_CLIENT:
			case USE_CLIENT_DELETE_OVER_CHANGE:
				serverModifications.remove(serverChange);
				DataObject copy = serverChange.createCopy(true); // copy current server state
				serverChange.rollbackChanges(); // Restore original state from Cache
				DataObjectUtil.modify(copy, serverChange); // create update that undoes all modifications on server
				newServerModifications.add(DataObjectUtil.copyModifications(copy, f)); // modify update so that it also contains client changes
				return;
			case USE_SERVER:
			case USE_SERVER_DELETE_OVER_CHANGE:
				f.setFailed(true);
				return;
		}
		throw new InternalUSMException(USMSessionManagementErrorCodes.ILLEGAL_CONFLICT_RESOLUTION_NUMBER1,
				"Illegal ConflictResolution " + conflictResolution);
	}
}
