/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.database.ox;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.util.SQLToolkit;
import com.openexchange.usm.util.Toolkit;

public class TableDefinition {

    final static String SESSION_TABLE_NAME = "USMSession";

    private static final Log LOG = LogFactory.getLog(TableDefinition.class);

    // Names of added tables
    public final static String[] TABLE_NAMES = { "USMSession", "usmIdMapping", "easUIDMapping", "USMDataStorage", "USMSessionFields" };

    public final static String[] SEQUENCE_TABLES_NAMES = { "sequence_usmSessionId", "sequence_usmIdMapping" };

    // SQL statements to create tables
    public final static String[] SQL_STATEMENTS = {
        "CREATE TABLE USMSession (cid INT4 UNSIGNED NOT NULL, id INT4 UNSIGNED NOT NULL, device VARCHAR(64) NOT NULL, protocol VARCHAR(32) NOT NULL, usmSessionId INT4 UNSIGNED NOT NULL, PRIMARY KEY (cid, usmSessionId), UNIQUE (cid, id, device, protocol)) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_bin",
        "CREATE TABLE usmIdMapping(cid INT4 UNSIGNED NOT NULL, usmSessionId INT4 UNSIGNED NOT NULL, longId VARCHAR(255) NOT NULL, shortId INT4 UNSIGNED NOT NULL, PRIMARY KEY (cid, usmSessionId, shortId), FOREIGN KEY(cid, usmSessionId) REFERENCES USMSession(cid, usmSessionId)) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_bin",
        "CREATE TABLE easUIDMapping(cid INT4 UNSIGNED NOT NULL, usmSessionId INT4 UNSIGNED NOT NULL, folderId VARCHAR(64) NOT NULL, longId VARCHAR(255) NOT NULL, shortId INT4 UNSIGNED NOT NULL, PRIMARY KEY (cid, usmSessionId, folderId, shortId), FOREIGN KEY(cid, usmSessionId) REFERENCES USMSession(cid, usmSessionId)) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_bin",
        "CREATE TABLE USMDataStorage (cid INT4 UNSIGNED NOT NULL, usmSessionId INT4 UNSIGNED NOT NULL, ObjectID VARCHAR(64) NOT NULL, SyncKey INT8 NOT NULL, Element LONGTEXT NOT NULL, PRIMARY KEY (cid, usmSessionId, objectId, syncKey), FOREIGN KEY(cid, usmSessionId) REFERENCES USMSession(cid, usmSessionId)) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_bin",
        "CREATE TABLE USMSessionFields (cid INT4 UNSIGNED NOT NULL, usmSessionId INT4 UNSIGNED NOT NULL, field VARCHAR(20) NOT NULL, value VARCHAR(32) NOT NULL, PRIMARY KEY (cid, usmSessionId, field), FOREIGN KEY(cid, usmSessionId) REFERENCES USMSession(cid, usmSessionId)) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_bin" };

    public final static String[] UUID_SQL_STATEMENTS = {
        "CREATE TABLE UUIDHeader(cid INT4 UNSIGNED NOT NULL, header VARCHAR(36) NOT NULL, PRIMARY KEY (header)) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_bin",
        "CREATE TABLE UUIDMapping (cid INT4 UNSIGNED NOT NULL, contentType INT4 UNSIGNED NOT NULL, objectId INT4 UNSIGNED NOT NULL, uuid BINARY(16) NOT NULL, PRIMARY KEY (cid, uuid), UNIQUE (cid, contentType, objectId)) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_bin" };

    public static void generateSequence(String sequenceName, Connection con, boolean isAutocommit, boolean createTable) throws USMStorageException, StorageAccessException {
        Statement stmt = null;
        try {
            try {
                stmt = con.createStatement();
                if (isAutocommit)
                    con.setAutoCommit(false);
                if (createTable)
                    stmt.execute(SQLToolkit.createSequenceStatement(sequenceName));
                // do not create stored procedures anymore, they will be deleted by USMDeleteStoredProceduresUpdateTaskV2 anyway
                //stmt.execute(SQLToolkit.dropSequenceProcedure(sequenceName));
                //stmt.execute(SQLToolkit.createSequenceProcedure(sequenceName));
                con.commit();
                LOG.info("Creation of sequence " + sequenceName + " successfully performed");
            } catch (SQLException e) {
                if (isAutocommit)
                    con.rollback();
                throw new USMStorageException(USMDatabaseOXErrorCodes.CREATE_SEQUENCE_ERROR, e.getMessage(), e);
            } finally {
                if (isAutocommit)
                    con.setAutoCommit(true);
                Toolkit.close(stmt);
            }
        } catch (SQLException e) {
            throw new USMStorageException(USMDatabaseOXErrorCodes.CREATE_SEQUENCE_ERROR_AUTOCOMMIT_SETTINGS, e.getMessage(), e);
        }
    }

}
