/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMResourceAccessException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.SessionInitializer;
import com.openexchange.usm.api.session.assets.ConflictResolution;
import com.openexchange.usm.api.session.assets.SessionInformation;
import com.openexchange.usm.api.session.assets.USMSessionID;
import com.openexchange.usm.api.session.storage.PersistentSessionDataStorage;
import com.openexchange.usm.util.BitSetEncoder;

/**
 * @author afe
 */
public class PersistentSessionData implements Serializable {

    private static final Log log = LogFactory.getLog(PersistentSessionData.class);

    // Determines in ms. how often for constantly active sessions the internal cache of persistent data is updated (1/hour)
    private final static long UPDATE_INTERVAL = 60L * 60L * 1000L;

    private final static String CONFLICT_RESOLUTION = "ConflictResolution";

    private static final String CONTENT_TYPE_PREFIX = "CT_";

    private static final String FIELD_PREFIX = "F_";

    private final SessionImpl session;

    // private final SessionInformation si;

    private volatile Map<ContentType, BitSet> _contentTypeFilter = new HashMap<ContentType, BitSet>();

    private volatile Map<String, String> _fields = new HashMap<String, String>();

    private volatile Map<String, String> _fieldView = Collections.unmodifiableMap(_fields);

    private ConflictResolution _conflictResolution = ConflictResolution.ERROR;

    private long _lastUpdateFromDB;

    /**
     * Initializes a new {@link PersistentSessionData}.
     * 
     * @param sessionImpl
     */
    public PersistentSessionData(SessionImpl sessionImpl) {
        session = sessionImpl;
    }

    public void initialize(SessionInitializer initializer, JSONObject configuration) throws USMException {
        boolean hadFields = readPersistentData(_contentTypeFilter, _fields);
        if (initializer != null) {
            if (!hadFields)
                initializer.initializeSession(session, configuration);
            else
                initializer.reinitializeSession(session, configuration);
        } else {
            if (!hadFields)
                setContentTypeFilter(session.getSessionManager().getContentTypeManager().getRegisteredContentTypes());
        }
    }

    private void updatePersistentDataFromDB() {
        Map<ContentType, BitSet> contentTypeFilter = new HashMap<ContentType, BitSet>();
        Map<String, String> fields = new HashMap<String, String>();
        try {
            readPersistentData(contentTypeFilter, fields);
            _contentTypeFilter = contentTypeFilter;
            _fields = fields;
            _fieldView = Collections.unmodifiableMap(_fields);
        } catch (StorageAccessException e) {
            throw new USMResourceAccessException(
                USMSessionManagementErrorCodes.PERSISTENT_SESSION_DB_ERROR_3,
                "DB Access Exception on reading persistent fields",
                e);
        } catch (USMStorageException e) {
            throw new USMResourceAccessException(
                USMSessionManagementErrorCodes.PERSISTENT_SESSION_DB_ERROR_4,
                "SQL Exception on reading persistent fields",
                e);
        }
    }

    /**
     * Check whether the session exists
     * 
     * @return
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    public boolean checkProperDBStorage() throws StorageAccessException, USMStorageException {
        return getStorage().sessionExists(session);
    }

    /**
     * Read persistent data from db
     * 
     * @param contentTypeFilter
     * @param fields
     * @return
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    private boolean readPersistentData(Map<ContentType, BitSet> contentTypeFilter, Map<String, String> fields) throws StorageAccessException, USMStorageException {
        boolean hadFields = false;
        Integer sessionID = getStorage().getSessionID(session);

        if (sessionID != null) {

            // the session was existing
            hadFields = true;
            session.setUniqueSessionID(sessionID);

            // initialize eventually existing fields
            Map<String, String> storedFields = getStorage().getFields(session);
            Iterator<String> keys = storedFields.keySet().iterator();
            while (keys.hasNext()) {
                String k = keys.next();
                String v = storedFields.get(k);
                readPersistentEntry(contentTypeFilter, fields, k, v);
            }
        } else {
            // the session was not existing in the db
            session.setUniqueSessionID(getStorage().insertSession(session));
        }
        _lastUpdateFromDB = System.currentTimeMillis();

        return hadFields;
    }

    private void readPersistentEntry(Map<ContentType, BitSet> contentTypeFilter, Map<String, String> fieldMap, String field, String value) {
        if (CONFLICT_RESOLUTION.equals(field)) {
            try {
                _conflictResolution = ConflictResolution.valueOf(value);
            } catch (IllegalArgumentException iae) {
                session.getSessionManager().getJournal().error(session + " Invalid ConflictResolution " + value + " in DB", iae);
            }
        } else if (field.startsWith(FIELD_PREFIX)) {
            fieldMap.put(field.substring(2), value);
        } else {
            if (field.startsWith(CONTENT_TYPE_PREFIX)) { // Check for ContentType
                String typeName = field.substring(CONTENT_TYPE_PREFIX.length());
                ContentType type = session.getSessionManager().getContentTypeManager().getContentType(typeName);
                if (type != null) {
                    BitSet fields = validateFieldFilter(type, BitSetEncoder.fromBase64(value));
                    contentTypeFilter.put(type, fields);
                    return;
                }
            }
            log.warn(session + " Unknown field " + field + " in DB");
        }
    }

    public ConflictResolution getConflictResolution() {
        return _conflictResolution;
    }

    public void setConflictResolution(ConflictResolution resolution) throws StorageAccessException, USMStorageException {
        if (resolution == null)
            throw new IllegalArgumentException("ConflictResolution not specified (null)");
        if (_conflictResolution != resolution) {
            insertOrUpdateDBField(CONFLICT_RESOLUTION, resolution.toString());
            _conflictResolution = resolution;
        }
    }

    public void setContentTypeFilter(ContentType... usedContentTypes) throws USMStorageException, StorageAccessException {
        for (Iterator<ContentType> i = _contentTypeFilter.keySet().iterator(); i.hasNext();) {
            removeContentTypeFilter(i, usedContentTypes);
        }
        for (ContentType newType : usedContentTypes) {
            if (!_contentTypeFilter.containsKey(newType)) {
                int fieldCount = newType.getFields().length;
                BitSet fields = new BitSet(fieldCount);
                fields.set(0, fieldCount);
                setFieldFilter(newType, fields);
            }
        }
    }

    private void removeContentTypeFilter(Iterator<ContentType> i, ContentType... usedContentTypes) throws USMStorageException, StorageAccessException {
        ContentType storedType = i.next();
        for (ContentType newType : usedContentTypes) {
            if (newType.equals(storedType))
                return;
        }
        removeContentTypeFilterFromDB(storedType);
        i.remove();
    }

    public void setFieldFilter(ContentType type, BitSet fieldSet) throws USMStorageException, StorageAccessException {
        BitSet fields = validateFieldFilter(type, fieldSet);
        if (fields == null || fields.isEmpty()) {
            if (_contentTypeFilter.containsKey(type)) {
                removeContentTypeFilterFromDB(type);
                _contentTypeFilter.remove(type);
            }
        } else {
            if (!fields.equals(_contentTypeFilter.get(type))) {
                insertOrUpdateDBField(CONTENT_TYPE_PREFIX + type.getID(), BitSetEncoder.toBase64(fields));
                _contentTypeFilter.put(type, fields);
            }
        }
    }

    private static BitSet validateFieldFilter(ContentType type, BitSet f) {
        if (f == null || f.isEmpty())
            return f;
        ContentTypeField[] fields = type.getFields();
        BitSet result = new BitSet();
        int limit = Math.min(fields.length, f.length());
        for (int i = 0; i < limit; i++) {
            if ((areNegativeFieldIDsAllowed(type) ? true : fields[i].getFieldID() >= 0) && f.get(i))
                result.set(i);
        }
        if (!result.isEmpty()) {
            result.set(0); // Make sure that the first two fields (id and folder_id) are always retrieved
            result.set(1);
        }
        return result;
    }

    private static boolean areNegativeFieldIDsAllowed(ContentType type) {
        return DefaultContentTypes.GROUPS_ID.equals(type.getID()) || DefaultContentTypes.RESOURCES_ID.equals(type.getID());
    }

    public void setField(String key, String value) throws USMStorageException, StorageAccessException {
        if (value == null || value.length() == 0) {
            if (_fields.containsKey(key)) {
                removeDBField(FIELD_PREFIX + key);
                _fields.remove(key);
            }
        } else {
            if (!value.equals(_fields.get(key))) {
                insertOrUpdateDBField(FIELD_PREFIX + key, value);
                _fields.put(key, value);
            }
        }
    }

    private void checkForPossibleDBUpdate() {
        if (System.currentTimeMillis() > _lastUpdateFromDB + UPDATE_INTERVAL)
            updatePersistentDataFromDB();
    }

    public String getField(String key) {
        checkForPossibleDBUpdate();
        String result = _fields.get(key);
        return result == null ? "" : result;
    }

    public Map<String, String> getPersistentFields() {
        checkForPossibleDBUpdate();
        return _fieldView;
    }

    private void removeContentTypeFilterFromDB(ContentType type) throws USMStorageException, StorageAccessException {
        removeDBField(CONTENT_TYPE_PREFIX + type.getID());
    }

    /**
     * Remove the specified field
     * 
     * @param key
     * @throws USMStorageException
     * @throws StorageAccessException
     */
    private void removeDBField(String key) throws USMStorageException, StorageAccessException {
        getStorage().removeField(session, key);
    }

    /**
     * Purge fields for the specified session
     * 
     * @throws USMStorageException
     * @throws StorageAccessException
     */
    public void removeAllDBFieldsForSession() throws USMStorageException, StorageAccessException {
        getStorage().removeAllFieldsForSession(session);
    }

    private void insertOrUpdateDBField(String key, String value) throws StorageAccessException, USMStorageException {
        getStorage().storeFieldValue(session, key, value);
    }

    public BitSet getFieldFilter(ContentType contentType) {
        return _contentTypeFilter.get(contentType);
    }

    public ContentType[] getContentTypes() {
        Set<ContentType> keySet = _contentTypeFilter.keySet();
        return keySet.toArray(new ContentType[keySet.size()]);
    }

    /**
     * Get the specified fields from DB
     * 
     * @param sessionManager
     * @param cid
     * @param id
     * @param protocol
     * @param device
     * @param field
     * @return
     */
    public static List<String> getFieldsFromDB(SessionManagerImpl sessionManager, int cid, int id, String protocol, String device, String field) {
        if (cid != 0)
            return getFieldsFromSchema(sessionManager, cid, cid, id, protocol, device, field, null);
        List<String> result = new ArrayList<String>();
        try {
            Set<Integer> remainingContextIDs = new HashSet<Integer>(sessionManager.getSessionDataStorage().getAllContextIDs());
            while (!remainingContextIDs.isEmpty()) {
                int context = remainingContextIDs.iterator().next();
                for (String s : getFieldsFromSchema(sessionManager, context, 0, id, protocol, device, field, remainingContextIDs))
                    result.add(s);
                for (int contextId : sessionManager.getSessionDataStorage().getContextsInSameSchema(context))
                    remainingContextIDs.remove(contextId);
            }
            return result;
        } catch (StorageAccessException e) {
            List<String> ret = new ArrayList<String>(1);
            ret.add("An error has occurred while determining all context ids:");
            ret.add(e.toString());
            return ret;
        }
    }

    /**
     * Get the specified fields from context
     * 
     * @param sessionManager
     * @param contextForDB
     * @param cid
     * @param id
     * @param protocol
     * @param device
     * @param field
     * @return
     */
    private static List<String> getFieldsFromSchema(SessionManagerImpl sessionManager, int contextForDB, int cid, int id, String protocol, String device, String field, Set<Integer> validContextIDs) {
        try {
            return sessionManager.getSessionDataStorage().getFieldsFromSchema(
                contextForDB,
                cid,
                id,
                protocol,
                device,
                field,
                validContextIDs);
        } catch (Exception e) {
            List<String> fields = new ArrayList<String>();
            fields.add(cid + ": Error while retrieving session fields: " + e.getMessage());
            fields.add(e.toString());
            return fields;
        }
    }

    /**
     * Update the specified fields for the specified context
     * 
     * @param cid
     * @param id
     * @param protocol
     * @param device
     * @param field
     * @param value
     * @return
     */
    public static List<String> updateFieldsInDB(SessionManagerImpl sessionManager, int cid, int id, String protocol, String device, String field, String value) {
        if (!isParameterSet(field))
            throw new IllegalArgumentException("Parameter 'field' is required");
        if (cid != 0)
            return updateFieldsInSchema(sessionManager, cid, cid, id, protocol, device, field, value, null);
        try {
            List<String> result = new ArrayList<String>();
            Set<Integer> remainingContextIDs = new HashSet<Integer>(sessionManager.getSessionDataStorage().getAllContextIDs());
            while (!remainingContextIDs.isEmpty()) {
                int context = remainingContextIDs.iterator().next();
                for (String s : updateFieldsInSchema(sessionManager, context, 0, id, protocol, device, field, value, remainingContextIDs))
                    result.add(s);
                for (int contextId : sessionManager.getSessionDataStorage().getContextsInSameSchema(context))
                    remainingContextIDs.remove(contextId);
            }
            return result;
        } catch (StorageAccessException e) {
            List<String> ret = new ArrayList<String>(2);
            ret.add("An error has occurred while determining all context ids:");
            ret.add(e.toString());
            return ret;
        }
    }

    /**
     * Updates the specified field for the specified context
     * 
     * @param sessionManager
     * @param contextForDB
     * @param cid
     * @param id
     * @param protocol
     * @param device
     * @param field
     * @param value
     * @param validContextIDs
     * @return
     */
    private static List<String> updateFieldsInSchema(SessionManagerImpl sessionManager, int contextForDB, int cid, int id, String protocol, String device, String field, String value, Set<Integer> validContextIDs) {
        try {
            Map<USMSessionID, SessionInformation> sessionIDs = sessionManager.getSessionDataStorage().updateFieldsInSchema(
                contextForDB,
                cid,
                id,
                protocol,
                device,
                field,
                value,
                validContextIDs);

            if (sessionIDs.isEmpty()) {
                List<String> ret = new ArrayList<String>(1);
                ret.add(contextForDB + ": >> No matching sessions <<");
                return ret;
            }

            for (SessionImpl session : sessionManager.getSessionStorage().getSessionList()) {
                if (sessionIDs.containsKey(new USMSessionID(session.getContextId(), session.getSessionId()))) {
                    PersistentSessionData persistentData = session.getPersistentSessionData();
                    if (isParameterSet(value))
                        persistentData._fields.put(field, value);
                    else
                        persistentData._fields.remove(field);
                }
            }

            List<String> fields = new ArrayList<String>();
            fields.add(cid + ": >> " + sessionIDs.size() + " sessions updated <<");
            return fields;

        } catch (Exception e) {
            List<String> fields = new ArrayList<String>(2);
            fields.add(cid + ": Error while updating session fields: " + e.getMessage());
            fields.add(e.toString());
            return fields;
        }
    }

    private static boolean isParameterSet(String param) {
        return param != null && param.length() > 0 && !"String".equals(param);
    }

    private PersistentSessionDataStorage getStorage() {
        return session.getSessionManager().getSessionDataStorage();
    }
}
