/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.util;

import org.apache.commons.logging.Log;
import com.openexchange.usm.api.session.Session;

/**
 * This class can be used by USM protocol implementations to check for clients that behave badly, i.e. perform too many unnecessary synchronizations.
 * If an unnecessary access is reported, an AccessCounter is created for that session. It will be removed if no unnecessary access was detected
 * within the test interval. If the limit of unnecessary accesses has been reached, blocking for that session will be enabled for the refusal interval.
 * 
 * @author afe
 *
 */
public class AccessLimiter {

	// Constant for the key under which the AccessCounter is stored as custom field in the USM Session 
	private final static String _USM_ACCESS_LIMITER = "usm.AccessLimiter";

	private final Log _journal;

	private final int _testInterval;
	private final int _refusalInterval;
	private final int _maxAccesses;

	public AccessLimiter(Log journal, int testInterval, int maxAccesses, int refusalInterval) {
		_journal = journal;
		_testInterval = testInterval;
		_refusalInterval = refusalInterval;
		_maxAccesses = maxAccesses;
	}

	/* (non-Javadoc)
	 * @see com.openexchange.usm.json.access.AccessLimiter#isUSMAccessDenied(com.openexchange.usm.api.session.Session)
	 */
	public boolean isUSMAccessDenied(Session session) {
		if (!isCheckEnabled())
			return false;
		Object o = session.getCustomProperty(_USM_ACCESS_LIMITER);
		if (!(o instanceof AccessCounter))
			return false;
		return ((AccessCounter) o).isAccessDenied(session);
	}

	public void removeCounter(Session session) {
		session.setCustomProperty(_USM_ACCESS_LIMITER, null);
	}

	/* (non-Javadoc)
	 * @see com.openexchange.usm.json.access.AccessLimiter#unnecessaryAccessEncountered(com.openexchange.usm.api.session.Session)
	 */
	public void badAccessEncountered(Session session) {
		if (isCheckEnabled())
			getAccessCounter(session).onUnnecessaryAccess(session);
	}

	private boolean isCheckEnabled() {
		return _testInterval > 0 && _refusalInterval > 0 && _maxAccesses > 0;
	}

	private AccessCounter getAccessCounter(Session session) {
		Object o = session.getCustomProperty(_USM_ACCESS_LIMITER);
		if (o instanceof AccessCounter)
			return (AccessCounter) o;
		AccessCounter p = new AccessCounter(this);
		session.setCustomProperty(_USM_ACCESS_LIMITER, p);
		return p;
	}

	public int getTestInterval() {
		return _testInterval;
	}

	public int getRefusalInterval() {
		return _refusalInterval;
	}

	public int getMaxAccesses() {
		return _maxAccesses;
	}

	public Log getJournal() {
		return _journal;
	}
}
