/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.api.configuration;

import com.openexchange.usm.api.exceptions.USMInvalidConfigurationException;

/**
 * Service that provides access to OX configuration or alternative configuration
 * 
 * <br>TODO Define which methods are provided, for now simple interface with no listeners, maybe adjust this to use OX config bundle ?
 * 
 * @author afe
 *
 */
public interface ConfigurationManager {

	/**
	 * Retrieves the given String property from the USM properties configuration file. If the property
	 * is not found, the defaultValue is returned. If warningIfNotPresent is set to true, a warning
	 * will be logged if the property is not set in the config file (the default value will be returned).
	 * 
	 * @param key
	 * @param defaultValue
	 * @param warningIfNotPresent 
	 * @throws USMInvalidConfigurationException
	 * @return
	 */
	String getProperty(String key, String defaultValue, boolean warningIfNotPresent) throws USMInvalidConfigurationException;

	/**
	 * Retrieves the given boolean property from the USM properties configuration file. If the property
	 * is not found, the defaultValue is returned. Boolean parsing is performed as in Boolean.parseBoolean().
	 * If warningIfNotPresent is set to true, a warning will be logged if the property is not set in the
	 * config file (the default value will be returned).
	 * 
	 * @param key
	 * @param defaultValue
	 * @param warningIfNotPresent 
	 * @throws USMInvalidConfigurationException
	 * @return
	 */
	boolean getProperty(String key, boolean defaultValue, boolean warningIfNotPresent) throws USMInvalidConfigurationException;

	/**
	 * Retrieves the given integer property from the USM properties configuration file. If the property
	 * is not found, the defaultValue is returned. Also, if the value in the property can not be parsed
	 * as an integer, this is logged as an error and USMInvalidConfigurationException is thrown. If warningIfNotPresent
	 * is set to true, a warning will be logged if the property is not set in the config file (the
	 * default value will be returned).
	 * 
	 * @param key
	 * @param defaultValue
	 * @param warningIfNotPresent 
	 * @throws USMInvalidConfigurationException
	 * @return
	 */
	int getProperty(String key, int defaultValue, boolean warningIfNotPresent) throws USMInvalidConfigurationException;

	/**
	 * Retrieves the given double property from the USM properties configuration file. If the property
	 * is not found, the defaultValue is returned. Also, if the value in the property can not be parsed
	 * as a double, this is logged as an error and USMInvalidConfigurationException is thrown. If warningIfNotPresent
	 * is set to true, a warning will be logged if the property is not set in the config file (the default
	 * value will be returned).
	 * 
	 * @param key
	 * @param defaultValue
	 * @param warningIfNotPresent
	 * @throws USMInvalidConfigurationException 
	 * @return
	 */
	double getProperty(String key, double defaultValue, boolean warningIfNotPresent) throws USMInvalidConfigurationException;
	
	
	/**
	 * Retrieves the given String property from the USM properties configuration file. If the property
	 * is not found, the defaultValue is returned. If warningIfNotPresent is set to true, a warning
	 * will be logged if the property is not set in the config file (the default value will be returned).
	 * 
	 * @param key
	 * @throws USMInvalidConfigurationException
	 * @return
	 */
	String getPropertyAsString(String key) throws USMInvalidConfigurationException;

	/**
	 * Retrieves the given boolean property from the USM properties configuration file. If the property
	 * is not found, the defaultValue is returned. Boolean parsing is performed as in Boolean.parseBoolean().
	 * If warningIfNotPresent is set to true, a warning will be logged if the property is not set in the
	 * config file (the default value will be returned).
	 * 
	 * @param key
	 * @throws USMInvalidConfigurationException
	 * @return
	 */
	boolean getPropertyAsBoolean (String key) throws USMInvalidConfigurationException;

	/**
	 * Retrieves the given integer property from the USM properties configuration file. If the property
	 * is not found, the defaultValue is returned. Also, if the value in the property can not be parsed
	 * as an integer, this is logged as an error and the default value is returned. If warningIfNotPresent
	 * is set to true, a warning will be logged if the property is not set in the config file (the
	 * default value will be returned).
	 * 
	 * @param key
	 * @throws USMInvalidConfigurationException
	 * @return
	 */
	int getPropertyAsInt (String key) throws USMInvalidConfigurationException;

	/**
	 * Retrieves the given double property from the USM properties configuration file. If the property
	 * is not found, the defaultValue is returned. Also, if the value in the property can not be parsed
	 * as a double, this is logged as an error and the default value is returned. If warningIfNotPresent
	 * is set to true, a warning will be logged if the property is not set in the config file (the default
	 * value will be returned).
	 * 
	 * @param key
	 * @throws USMInvalidConfigurationException 
	 * @return
	 */
	double getPropertyAsDouble (String key) throws USMInvalidConfigurationException;
}
