/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.sync;

import java.util.BitSet;
import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.contenttypes.FolderConstants;
import com.openexchange.usm.api.contenttypes.FolderContentType;
import com.openexchange.usm.api.contenttypes.FolderContentTypeTransferHandler;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.DataObjectFilter;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.impl.SessionImpl;

public class FolderHierarchyStorage implements ContentSyncerStorage {

    private final SessionImpl _session;

    private final DataObjectFilter _filter;

    private final boolean _storeResult;

    private final String[] _rootFolders;

    private final long _oldTimestamp;
    
    private final long _syncStartTimestamp;

    private final OXDataCacheID _cacheID;

    public FolderHierarchyStorage(SessionImpl session, DataObjectFilter filter, boolean storeResult, long oldTimestamp, String... rootFolders) {
        _session = session;
        _filter = filter;
        _storeResult = storeResult;
        _oldTimestamp = (filter != null && filter.getOriginalSyncID() != 0L) ? filter.getOriginalSyncID() : oldTimestamp;
        _rootFolders = rootFolders;
        _syncStartTimestamp = System.currentTimeMillis();
        _cacheID = session.getFolderHierarchyCacheID();
    }

    public boolean getCurrentServerData(DataObjectSet resultSet) throws USMException {
        FolderContentType contentType = _session.getSessionManager().getFolderContentType();
        FolderContentTypeTransferHandler handler = contentType.getTransferHandler();
        BitSet bits = _session.getValidFilter(contentType);
        if (_rootFolders.length == 0)
            addToFolderSet(resultSet, handler, handler.readRootFolders(_session, bits));
        else {
            for (String id : _rootFolders) {
                Folder f = (Folder) resultSet.get(id);
                if (f == null) { // Not already read as sub-folder of another root folder ?
                    f = contentType.newDataObject(_session);
                    f.setID(id);
                    handler.readDataObject(f, bits);
                    addToFolderSet(resultSet, handler, f);
                }
                if (id.equals("0")) {
                    // If the OX root folder is in the list of root folders, remove it from the result list (it should not be visible
                    // itself, and if present, other root folders may be listed below it, which would be wrong)
                    resultSet.remove(id);
                } else {
                    f.setParentFolderID("0"); // Mark as root folder
                    f.commitChanges();
                }
            }
        }
        return false;
    }

    public long storeServerData(long timestamp, DataObjectSet data, boolean updatesPending) throws USMException {
        return _storeResult ? _session.storeCachedFolders(_oldTimestamp, timestamp, data, updatesPending, _syncStartTimestamp) : 0L;
    }

    public OXDataCacheID getCacheID() {
        return _cacheID;
    }

    @Override
    public String toString() {
        return _session + " FolderHierarchy(" + _storeResult + ')';
    }

    private void addToFolderSet(DataObjectSet folderSet, FolderContentTypeTransferHandler handler, Folder f) throws USMException {
        if (!folderSet.contains(f.getID())) {
            completeFolderInformation(f);
            if (_filter == null || _filter.accept(f)) {
                folderSet.add(f);
                if (f.hasSubFolders())
                    addToFolderSet(folderSet, handler, handler.readFolderContent(f));
            }
        }
    }

    private void addToFolderSet(DataObjectSet folderSet, FolderContentTypeTransferHandler handler, Folder... folders) throws USMException {
        for (Folder f : folders)
            addToFolderSet(folderSet, handler, f);
    }

    private void completeFolderInformation(Folder f) {
        if (f.getElementsContentType() == null) {
            ContentType t = _session.getSessionManager().getContentTypeManager().getContentType(f.getElementsContentTypeID());
            if (t != null)
                f.setElementsContentType(t);
        }
        f.commitChanges();
    }

    public DataObject findMatchingClientObjectForServerObject(DataObjectSet clientDataObjects, DataObject serverObject) {
        if (!_session.getSessionInitializer().objectsSynchronizedByUUID())
            return null;
        Object v = serverObject.getFieldContent(FolderConstants.STANDARD_FOLDER_TYPE);
        if (!(v instanceof Number) || ((Number) v).intValue() == 0)
            return null;
        //FIXME: Issues with Unified Mail - for more info in relevant e-mail message
        /*for (DataObject i : clientDataObjects) {
            if (v.equals(i.getFieldContent(FolderConstants.STANDARD_FOLDER_TYPE))) {
                _session.getSessionManager().getJournal().warn(
                    _session + " Standard folder (type " + v + ") changed OX id: '" + i.getID() + "' -> '" + serverObject.getID() + '\'');
              return i;
            }
        }*/
        return null;
    }
}
